package hu.iqsoft.otp.webshop;

import hu.iqsoft.otp.webshop.factory.WSAnswerFactory;
import hu.iqsoft.otp.webshop.mwaccess.MWAccessBindingStub;
import hu.iqsoft.otp.webshop.mwaccess.WorkflowState;
import hu.iqsoft.otp.webshop.util.SignatureUtils;
import hu.iqsoft.otp.webshop.util.TransactionLogger;
import hu.iqsoft.otp.webshop.util.WebShopXmlUtils;
import hu.iqsoft.otp.webshop.model.WebShopFizetesAdatokLista;
import hu.iqsoft.otp.webshop.model.WebShopFizetesValasz;
import hu.iqsoft.otp.webshop.model.WebShopTranzAzon;
import hu.iqsoft.otp.workflow.wlpi.WResponse;
import hu.iqsoft.otp.workflow.wlpi.WResponseByDOM;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.rmi.RemoteException;
import java.security.Security;
import java.util.Date;
import java.util.Properties;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.axis.AxisFault;
import org.apache.axis.AxisProperties;
import org.apache.log4j.Category;
import org.apache.log4j.PropertyConfigurator;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;

/**
 * Kliens oldali WebShop szerver.
 *
 * A WebShop-ok rszre ksztett, WebService felleten hvhat web-alkalmazs
 * rsze. A web-alkalmazst lokkisal, WebShop-onknt kell telepteni.Ezek az
 * alkalmazsok hvjk meg az OTP middleware rendszernek megfelel WebShop
 * folyamatait: - ping - tranzakci azonost generls - hromszerepls
 * fizetsi folyamat indtsa - ktszerepls fizetsi folyamat indtsa -
 * tranzakci adatok, tranzakci sttusz lekrdezse
 *
 * A fenti szolgltatsok kzvetlen mdon, az gynevezett OTP MWAccess felleten
 * is meghvhatak, de ott lnyegesen sszetettebb feladat hrul a WebShop
 * kliens oldali fejesztkre. A kliens oldali WebShop szerver az albbi
 * funkcik vgrehajtsval knnyti a fejlsztst: - nem kell folyamatokat
 * indt xml-eket sszelltani s nem kell vlasz xml- eket rtelmezni.
 * Egyszeren mezszinten kell megadni a bemen adatokat, s ugyancsak
 * mezszinten rhetek e a vlasz adatai. - az alkalmazs automatikusan
 * legenerlja a digitlis alrst azoknl a mveleteknl, ahol ez szksges. -
 * automatikusan naplzsra kerlnek a kommunikci elemei: a kapcsoldsi
 * paramterek, a bejv s kimen SOAP krsek, a folyamat input- s answer
 * xml- jei. A naplzs rszletessge konfigurlhat.
 *
 * Digitlis alrs a kvetkez esetekben kpzdik: -
 * tranzakcioStatuszLekerdezes - fizetesiTranzakcio -
 * fizetesiTranzakcioKetszereplos
 *
 * Az egyes naplzsi szinteken az albbi esemnyek s adatok naplzdnak: Debug
 * szint zenetek: - Krs feldolgozs indulsa, befejezdse - OTP Middleware
 * fel men s onnan rkez SOAP krsek tartalma - OTP Middleware fel men s
 * onnan rkez workflow xml-ek (indt s vlasz) tartalma
 *
 * Info szint zenetek: - MW kapcsoldsi paramterek hvsonknt - Indtott
 * mveletek OTP middleware ltal generlt folyamat-azonosti. Ezekkel az
 * azonostkkal kerlnek az OTP rendszerben naplzsra a tranzakci adatai, s
 * ezzel az azonostval lehet annak lefutst ellenrizni.
 *
 * Warning szint zenetek: - Hibval lefutott krsek, melyek elindultak az OTP
 * middleware fel. Ezek szablyos SOAP hibazenetknt kerlnek tovbbtsra a
 * kliens fel. Plda: hibs a Middleware szerver cme / a szerver nem elrhet /
 * a szerver adott szolgltatsa nem elrhet, stb.
 *
 * Fatal szint zenetek: - Olyan hibk, melyek miatt az OTP Middleware szerver
 * fel el sem indulhatott a krs. Pldul hinyos teleptsbl add hibk.
 *
 * @version 1.2
 * @author Bodnr Imre (c) IQSYS Rt.
 */
public class WebShopService extends MWAccessBindingStub {

    public final static String RESOURCE_BUNDLE = "otp_webshop_client.properties";

    public final static String PROPERTY_PRIVATEKEYFILE = "otp.webshop.PRIVATE_KEY_FILE";

    public final static String PROPERTY_OTPMWSERVERURL = "otp.webshop.OTPMW_SERVER_URL";

    public final static String PROPERTY_HTTPSPROXYHOST = "otp.webshop.client.HTTPS_PROXYHOST";

    public final static String PROPERTY_HTTPSPROXYPORT = "otp.webshop.client.HTTPS_PROXYPORT";

    public final static String PROPERTY_HTTPSPROXYUSER = "otp.webshop.client.HTTPS_PROXYUSER";

    public final static String PROPERTY_HTTPSPROXYPASSWORD = "otp.webshop.client.HTTPS_PROXYPASSWORD";

    public final static String PROPERTY_TRANSACTIONLOGDIR = "otp.webshop.TRANSACTION_LOG_DIR";

    public final static String PROPERTY_MULTIPRIVATEKEY_PREFIX = "otp.webshop.PRIVATE_KEY_FILE_";

    public final static String PROPERTY_MULTITRANSACTIONLOG_PREFIX = "otp.webshop.TRANSACTION_LOG_DIR_";

    public final static String SERVICEERROR_RESOURCE_NOTFOUND = "EROFORRASLEIROFILE_NEMLETEZIK";

    public final static String SERVICEERROR_PRIVATEKEYFILE_NOTFOUND = "PRIVATKULCSFILE_NEMLETEZIK";

    public final static String LOG_FATALERROR = "FATALISHIBA";

    public final static String SYSTEM_FATALERROR = "RENDSZERHIBA";

    public final static String SERVERERROR_INFO = "SZERVERHIBA";

    protected final static String DEFAULT_DEVIZANEM = "HUF";

    /* jrakldsek maximlis szma */
    protected final static int RESENDCOUNT = 20;
    
    /* jrakldsek kzti ksleltetsi id */
    protected final static int RESENDDELAY = 500;
    
    /* jrakldsek csak akkor lehetsgesek, ha a lenti szvegek valamelyike
     * szerepel az elutastshoz tartoz kivtelben. */
    protected final static String[] RESEND_ERRORPATTERN = 
    	{"Maximum workflow number is reached"};
    
    protected final static WSAnswerFactory ANSWERFACTORY = new WSAnswerFactory();

    protected static Category category = null;

    static {
        try {

            InputStream is = WebShopService.class.getClassLoader()
                    .getResourceAsStream(RESOURCE_BUNDLE);
            if (is == null) {
                throw new AxisFault(SERVICEERROR_RESOURCE_NOTFOUND,
                        RESOURCE_BUNDLE, null, null);
            }
            Properties props = new Properties();
            props.load(is);
            is.close();

            // do the first initialization of the file
            PropertyConfigurator.configure(props);

            System.out.println("OTPWebShopService naplozas inicializalva.");
        } catch (Exception ex) {
            // log4j config file not found. just print a message to standard
            // error.
            System.out
                    .println("Hiba az OTPWebShopService loggolas inicializalasa kozben.\n"
                            + ex.getMessage());
        }

        category = Category.getInstance("WebShopClient");
        synchronized (MWAccessBindingStub.class) {
            if (MWAccessBindingStub.category == null) {
                MWAccessBindingStub.category = category;
            }
        }

        /*
         * SSL kapcsolatok kialaktshoz szksges inicializlsok
         */
        // A Sun's reference implementci URL handler hasznlata a "https"
        // protokollhoz.
        System.setProperty("java.protocol.handler.pkgs",
                "com.sun.net.ssl.internal.www.protocol");

        // Dinamikusan regisztrljuk a sun's ssl provider-t
        Security.addProvider(new com.sun.net.ssl.internal.ssl.Provider());

        /*
         * Az AXIS-ban beregisztrlunk egy olyan SSLSocketFactorty osztlyt,
         * amely nem hitelesti a klienst.
         */
        AxisProperties.setClassDefault(
                org.apache.axis.components.net.SecureSocketFactory.class,
                "hu.iqsoft.otp.webshop.util.NoClientAuthJSSESocketFactory");

    }

    private static Properties webShopClientProperties;

    /**
     * Konstruktor. Beolvassa a property fjlot s az alapjn inicializlja a mw
     * kapcsolatot s a ssl belltsokat
     */
    public WebShopService() throws org.apache.axis.AxisFault {
        this(null);
        try {
            category.debug("OTPWebShopService pldnyosts...");

            Properties property = getWebShopProperties();

            // A Sun's reference implementci URL handler hasznlata a "https"
            // protokollhoz.
            System.setProperty("java.protocol.handler.pkgs",
                    "com.sun.net.ssl.internal.www.protocol");

            // Dinamikusan regisztrljuk a sun's ssl provider-t
            Security.addProvider(new com.sun.net.ssl.internal.ssl.Provider());

            String mwURL = property.getProperty(PROPERTY_OTPMWSERVERURL);
            // mwURL = new URL("https://localhost:7002/mwaccess/mwaccess");
            super.cachedEndpoint = new URL(mwURL);
            category.info("OTPMW szerver url: " + mwURL);

            String proxyHost = property.getProperty(PROPERTY_HTTPSPROXYHOST);
            if ((proxyHost != null) && (proxyHost.trim().length() != 0)) {
                System.setProperty("https.proxyHost", proxyHost);
                category.info("Kliens https proxy host: " + proxyHost);
            }

            String proxyPort = property.getProperty(PROPERTY_HTTPSPROXYPORT);
            if ((proxyPort != null) && (proxyPort.trim().length() != 0)) {
                System.setProperty("https.proxyPort", proxyPort);
                category.info("Kliens https proxy port: " + proxyPort);
            }

            String proxyUser = property.getProperty(PROPERTY_HTTPSPROXYUSER);
            if ((proxyUser != null) && (proxyUser.trim().length() != 0)) {
                System.setProperty("https.proxyUser", proxyUser);
                category.info("Kliens https proxy user: " + proxyUser);
            }

            String proxyPassword = property
                    .getProperty(PROPERTY_HTTPSPROXYPASSWORD);
            if ((proxyPassword != null) && (proxyPassword.trim().length() != 0)) {
                System.setProperty("https.proxyPassword", proxyPassword);
                category.info("Kliens https proxy password: ******");
            }

            category.debug("OTPWebShopService pldnyosts befejezve...");
        } catch (Exception e) {
            category.fatal(LOG_FATALERROR, e);
            e.printStackTrace();
            throw new AxisFault(LOG_FATALERROR, e.getMessage(), null, null);
        }
    }

    public WebShopService(javax.xml.rpc.Service service)
            throws org.apache.axis.AxisFault {
        super(service);
    }

    /**
     * A vlasz xml-t vlasz objektumm alaktja
     */
    protected WResponse getResponse(byte[] response, String worfklow)
            throws Exception {
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        DocumentBuilder db = dbf.newDocumentBuilder();
        Document dom = db.parse(new InputSource(new ByteArrayInputStream(
                response)));
        WResponse result = new WResponseByDOM(dom.getDocumentElement(),
                worfklow);
        return result;
    }

    /**
     * Kiolvassa a vlasz objektumbl a hibazenetek listjt, s szkzzel
     * elvlasztva sszefzi ket.
     */
    protected String getErrorsAsList(WResponse response) {
        StringBuffer result = new StringBuffer("");

        for (int i = 0; i < response.getErrors().size(); i++) {
            result.append((String) response.getErrors().get(i));
            if (i < response.getErrors().size() - 1) {
                result.append(" ");
            }
        }

        return result.toString();
    }

    /**
     * A WebShopService property fjl betltse
     */
    protected static Properties getWebShopProperties() throws AxisFault {
        if (webShopClientProperties != null) {
            return webShopClientProperties;
        }
        try {
            // lehetne class.getResource is, de az URL-t ad vissza
            InputStream is = WebShopService.class.getClassLoader()
                    .getResourceAsStream(RESOURCE_BUNDLE);
            if (is == null) {
                throw new AxisFault(SERVICEERROR_RESOURCE_NOTFOUND,
                    RESOURCE_BUNDLE, null, null);
            }
            Properties props = new Properties();
            props.load(is);
            is.close();

            // Troljuk a beolvasott properties-t (singleton),
            // ha csak ellenkezo utasitas nincs
            if (props.getProperty("otp.webshop.debug.ALWAYS_REFRESH_PROPERTIES") == null) {
                webShopClientProperties = props;
            }
            return props;
        } catch (IOException ioe) {
            throw new AxisFault(SERVICEERROR_RESOURCE_NOTFOUND, ioe);
        }
    }

    /**
     * Egy adott fizetsi tranzakcihoz tartoz privt kulcs llomny elrsi
     * tvonalnak belltsa a konfigurcis paramterek alapjn. Ha adott
     * posId (bolt azonost) esetn a konfigurcis llomny tartalmaz
     * otp.webshop.PRIVATE_KEY_posId=[elrsi t] bejegyzst, akkor az eljrs
     * ezt a bejegyzst adja vlaszul. Egybknt a
     * otp.webshop.PRIVATE_KEY_FILE=[elrsi t] bejegyzsben szereplt. Az els
     * mdszerrel lehet multishop-ot kialaktani, vagyis olyan WebShop boolt
     * oldali szervert, amely tbb bolt krst is ki tudja szolglni, s
     * boltonknt (szksgszeren) eltr privt kulcs alapjn trtnik a
     * digitlis alrs.
     *
     * @param properties
     *            A shop-hoz tartoz konfigurcis belltsok
     * @param posId
     *            A tranzakcit indt shop azonostja
     * @return A megadott shop-hoz tartoz privt kulcs elrsi tvonala
     */
    protected String getPrivKeyFileName(Properties properties, String posId) {
        String privKeyFileName = properties.getProperty(
                PROPERTY_MULTIPRIVATEKEY_PREFIX + posId, null);

        if (privKeyFileName == null || privKeyFileName.trim().length() == 0) {
            privKeyFileName = properties.getProperty(PROPERTY_PRIVATEKEYFILE);
        }

        return privKeyFileName;
    }

    /**
     * Egy adott fizetsi tranzakcihoz tartoz tranzakcis napl llomny
     * elrsi tvonalnak belltsa a konfigurcis paramterek alapjn. Ha
     * adott posId (bolt azonost) esetn a konfigurcis llomny tartalmaz
     * otp.webshop.TRANSACTION_LOG_DIR_posId=[elrsi t] bejegyzst, akkor az
     * eljrs ezt a bejegyzst adja vlaszul. Egybknt a
     * otp.webshop.otp.webshop.TRANSACTION_LOG_DIR=[elrsi t] bejegyzsben
     * szereplt elrsi tvonalat. Az els mdszerrel lehet multishop-ot
     * kialaktani, vagyis olyan WebShop bolt oldali szervert, amely tbb bolt
     * krst is ki tudja szolglni, s boltonknt ms knyvtrba trtnik a
     * tranzakcik naplzsa.
     *
     * @param properties
     *            A shop-hoz tartoz konfigurcis belltsok
     * @param posId
     *            A tranzakcit indt shop azonostja
     * @return A megadott shop-hoz tartoz tranzakcis napl llomnyok
     *         knyvtrnak elrsi tvonala
     */
    protected String getTranLogDir(Properties properties, String posId) {
        String tranLogDir = properties.getProperty(
                PROPERTY_MULTITRANSACTIONLOG_PREFIX + posId, null);

        if (tranLogDir == null || tranLogDir.trim().length() == 0) {
            tranLogDir = properties.getProperty(PROPERTY_TRANSACTIONLOGDIR);
        }

        return tranLogDir;
    }

    /**
     * Ping MWAccess service meghvsa. Nem indul mw folyamat! Digitlis alrs
     * nem kpzdik.
     */
    public void ping() throws AxisFault {
        try {
            category.debug("ping indul...");
            super.ping();
            category.debug("ping befejezdtt.");
        } catch (Exception e) {
            category.warn(SERVERERROR_INFO, e);
            throw new AxisFault("", e.getMessage(), null, null);
        }
    }

    /**
     * Tranzakci indtsa. 
     * Ha a Bank tlterhels miatt elutastja a krst, automatikus
     * jraklds trtnik maximum RESENDCOUNT darabszmban, 
     * RESENDDELAY ezredmsodperces ksleltetssel.
     */
    public WorkflowState startWorkflowSynch(String workflowName, String inputXml) 
    	throws RemoteException {
		
    	WorkflowState workflowState = null;
    	int retryCount = 0;
    	do {
    		try {
    			workflowState = super.startWorkflowSynch(workflowName, inputXml);
    			break;
    		}
    		catch (RemoteException e) {
    			if ((e instanceof AxisFault) 
    				&& (retryCount < RESENDCOUNT)) {
    				
    				AxisFault axisFault = (AxisFault)e;
    				boolean resendAllowed = false;
    				for (int i=0; i<RESEND_ERRORPATTERN.length && !resendAllowed; i++) {
    					resendAllowed = (axisFault.getFaultReason().indexOf(RESEND_ERRORPATTERN[i]) > -1);	
    				}
    				
    				if (!resendAllowed) throw e;
    				
    				try {
    					Thread.sleep(RESENDDELAY);
    				} catch (InterruptedException ie) {};
    				category.debug("SOAP uzenet ismetelt elkuldese [" + workflowName + "]");
    			}
    			else
    				throw e;
    		}
    	} while (retryCount++ < RESENDCOUNT);
    	
    	return workflowState;
    }

    /**
     * WEBSHOPTRANZAZONGENERALAS folyamat szinkron indtsa Digitlis alrs
     * nem kpzdik.
     */
    public WebShopTranzAzon tranzakcioAzonositoGeneralas(String posId)
            throws AxisFault {
        try {
            WebShopTranzAzon result = null;

            category.debug("tranzakcioAzonositoGeneralas indul...");

            Document dom = WebShopXmlUtils.getRequestSkeleton();

            Node templateNameText = dom
                    .createTextNode(WebShopXmlUtils.TEMPLATENAME_TAGNAME);
            templateNameText.setNodeValue("WEBSHOPTRANZAZONGENERALAS");

            Node templateName = dom.getDocumentElement().getElementsByTagName(
                    WebShopXmlUtils.TEMPLATENAME_TAGNAME).item(0);
            templateName.appendChild(templateNameText);

            Node variables = dom.getDocumentElement().getElementsByTagName(
                    WebShopXmlUtils.VARIABLES_TAGNAME).item(0);

            WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.POSID,
                    posId);

            // alrs kiszmtsa s paramterknt beszrsa
            byte[] pkcs8PrivateKey = SignatureUtils
                    .loadPrivateKey(getPrivKeyFileName(getWebShopProperties(),
                            posId));

            String[] fields = { posId };
            byte[] alairasBytes = SignatureUtils.generateSignature(
                    SignatureUtils.getByteArray(fields), pkcs8PrivateKey);

            String alairas = SignatureUtils.toHex(alairasBytes, 0,
                    (int) alairasBytes.length);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.CLIENTSIGNATURE, alairas);

            // a folyamat indtsa loggozssal
            String domString = WebShopXmlUtils.DOMToXMLString(dom);
            category.debug("tranzakcioAzonositoGeneralas keres:\n" + domString);

            WorkflowState state = startWorkflowSynch(
                    "WEBSHOPTRANZAZONGENERALAS", WebShopXmlUtils
                            .DOMToXMLString(dom));

            category.info("tranzakcioAzonositoGeneralas folyamat azonosio: "
                    + state.getInstanceId());

            System.out.println("Start: WEBSHOPTRANZAZONGENERALAS with "
                    + state.getInstanceId() + " at " + new Date());

            WResponse response = getResponse(state.getResult(),
                    "WEBSHOPTRANZAZONGENERALAS");

            // a folyamat vlasznak loggoza
            domString = WebShopXmlUtils.DOMToXMLString(response
                    .getOriginalResponseDOM().getOwnerDocument());
            category
                    .debug("tranzakcioAzonositoGeneralas valasz:\n" + domString);

            response.acquire(false, ANSWERFACTORY);

            if (response.isSuccessful()) {
                result = (WebShopTranzAzon) response.getAnswer().getData();
                result.setValaszKod("SIKER");
            } else {
                result = new WebShopTranzAzon();
                result.setValaszKod(getErrorsAsList(response));
            }

            category
                    .debug("tranzakcioAzonositoGeneralas sikeresen befejezodott.");

            return result;

        } catch (Exception e) {
            e.printStackTrace();
            category.warn(SERVERERROR_INFO, e);
            throw new AxisFault("", e);
        }

    }

    /**
     * WEBSHOPTRANZAKCIOLEKERDEZES folyamat szinkron indtsa Digitlis alrs
     * kpzdik.
     */
    public WebShopFizetesAdatokLista tranzakcioStatuszLekerdezes(String posId,
            String azonosito, String maxRekordSzam, String idoszakEleje,
            String idoszakVege) throws AxisFault {
        try {
            WebShopFizetesAdatokLista result;

            category.debug("tranzakcioStatuszLekerdezes indul.");

            Document dom = WebShopXmlUtils.getRequestSkeleton();

            Node templateNameText = dom
                    .createTextNode(WebShopXmlUtils.TEMPLATENAME_TAGNAME);
            templateNameText.setNodeValue("WEBSHOPTRANZAKCIOLEKERDEZES");

            Node templateName = dom.getDocumentElement().getElementsByTagName(
                    WebShopXmlUtils.TEMPLATENAME_TAGNAME).item(0);
            templateName.appendChild(templateNameText);

            Node variables = dom.getDocumentElement().getElementsByTagName(
                    WebShopXmlUtils.VARIABLES_TAGNAME).item(0);

            /* paramterek beillesztse */
            WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.POSID,
                    posId);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.TRANSACTIONID, azonosito);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.QUERYMAXRECORDS, maxRekordSzam);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.QUERYSTARTDATE, idoszakEleje);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.QUERYENDDATE, idoszakVege);

            // alrs kiszmtsa s paramterknt beszrsa
            byte[] pkcs8PrivateKey = SignatureUtils
                    .loadPrivateKey(getPrivKeyFileName(getWebShopProperties(),
                            posId));
            String[] fields = { posId, azonosito, maxRekordSzam, idoszakEleje,
                    idoszakVege };
            byte[] autorizaciosKodBytes = SignatureUtils.generateSignature(
                    SignatureUtils.getByteArray(fields), pkcs8PrivateKey);

            String autorizaciosKod = SignatureUtils.toHex(autorizaciosKodBytes,
                    0, (int) autorizaciosKodBytes.length);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.CLIENTSIGNATURE, autorizaciosKod);

            // a folyamat indtsa loggozssal
            String domString = WebShopXmlUtils.DOMToXMLString(dom);
            category.debug("tranzakcioStatuszLekerdezes keres:\n" + domString);

            WorkflowState state = startWorkflowSynch(
                    "WEBSHOPTRANZAKCIOLEKERDEZES", WebShopXmlUtils
                            .DOMToXMLString(dom));

            category.info("tranzakcioStatuszLekerdezes folyamat azonosio: "
                    + state.getInstanceId());
            System.out.println("Finished: WEBSHOPTRANZAKCIOLEKERDEZES with "
                    + state.getInstanceId() + " at " + new Date());

            WResponse response = getResponse(state.getResult(),
                    "WEBSHOPTRANZAKCIOLEKERDEZES");

            // a folyamat vlasznak loggoza
            domString = WebShopXmlUtils.DOMToXMLString(response
                    .getOriginalResponseDOM().getOwnerDocument());
            category.debug("tranzakcioStatuszLekerdezes valasz:\n" + domString);

            response.acquire(false, ANSWERFACTORY);

            if (response.isSuccessful()) {
                result = (WebShopFizetesAdatokLista) response.getAnswer()
                        .getData();
                result.setValaszKod("SIKER");
            } else {
                result = new WebShopFizetesAdatokLista();
                result.setValaszKod(getErrorsAsList(response));
            }

            result.setPosId(posId);

            category
                    .debug("tranzakcioStatuszLekerdezes sikeresen befejezodott.");

            return result;

        } catch (Exception e) {
            // e.printStackTrace();
            category.warn(SERVERERROR_INFO, e);
            throw new AxisFault("", e.getMessage(), null, null);
        }

    }

    /**
     * Hromszerepls fizetsi folyamat (WEBSHOPFIZETES) szinkron indtsa
     * Az, hogy egy- vagy ktlpcss fizets trtnik-e, a bolt banki oldal
     * trolt alaprtelmezett belltstl fgg.
     * Digitlis alrs kpzdik.
     *
     * @param posId
     *            webshop azonost
     * @param tranzakcioAzonosito
     *            fizetsi tranzakci azonost
     * @param osszeg
     *            fizetend sszeg, (num, max. 13+2), opcionlis tizedesponttal.
     *            <br />
     *            Nulla is lehet, ha a regisztraltUgyfelId paramter ki van
     *            tltve, s az ugyfelRegisztracioKell rtke igaz. gy kell
     *            ugyanis jelezni azt, hogy nem tnyleges vsrlsi tranzakcit
     *            kell indtani, hanem egy gyfl regisztrlst, vagyis az
     *            gyfl krtyaadatainak bekrst s eltrolst a banki
     *            oldalon.
     * @param devizanem
     *            fizetend devizanem
     * @param nyelvkod
     *            a megjelentend vev oldali fellet nyelve
     * @param nevKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            nevt
     * @param orszagKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "orszg rszt"
     * @param megyeKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "megye rszt"
     * @param telepulesKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "telepls rszt"
     * @param iranyitoszamKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "irnytszm rszt"
     * @param utcaHazszamKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "utca/hzszm rszt"
     * @param mailCimKell
     *            a megjelentend vev oldali felleten be kellkrni a vev
     *            e-mail cmt
     * @param kozlemenyKell
     *            a megjelentend vev oldali felleten fel kell knlni a
     *            kzlemny megadsnak lehetsgt
     * @param vevoVisszaigazolasKell
     *            a tranzakci eredmnyt a vev oldalon meg kell jelenteni
     *            (azaz nem a backURL-re kell irnytani)
     * @param ugyfelRegisztracioKell
     *            ha a regisztraltUgyfelId rtke nem res, akkor megadja, hogy
     *            a megadott azonost jonnan regisztrland-e, vagy mr
     *            regisztrlsra kerlt az OTP Internetes Fizet felletn.
     *            Elbbi esetben a kliens oldali bngszben olyan fizet oldal
     *            fog megjelenni, melyen meg kell adni az azonosthoz tartoz
     *            jelszt, illetve a krtyaadatokat. Utbbi esetben csak az
     *            azonosthoz tartoz jelsz kerl beolvassra az rtestsi
     *            cmen kvl. Ha a regisztraltUgyfelId rtke res, a pamarter
     *            rtke nem kerl felhasznlsra.
     * @param regisztraltUgyfelId
     *            az OTP fizetfelleten regisztrland vagy regisztrlt gyfl
     *            azonost kdja.
     * @param shopMegjegyzes
     *            a webshop megjegyzse a tranzakcihoz a vev rszre
     * @param backURL
     *            a tranzakci vgrehajtsa utn erre az internet cmre kell
     *            irnytani a vev oldalon az gyfelet (ha a
     *            vevoVisszaigazolasKell hamis)
     * @param ketlepcsosFizetes
     * 			  megadja, hogy ktlpcss fizets indtand-e.
     *            True rtk esetn a fizetsi tranzakci ktlpcss lesz, 
     *            azaz a terhelend sszeg csupn zrolsra kerl, 
     *            s gy is marad a bolt ltal indtott lezr tranzakci 
     *            indtsig avagy a zrols elvlsig.
     *            Az alaprtelmezett (res) rtk a false rtket jelli, teht
     *            a szokvnyos egylpcss azonnali fizets trtnik.       
     *
     * @return a tranzakci adatai, mely tartalmazza a webshop ltal megadott
     *         fizetsi adatokat, a vev ltal megadott adatokat, s a
     *         tranzakci eredmnyt a tejests idejvel s a POS autorizcis
     *         kddal egytt.
     */
    public WebShopFizetesAdatokLista fizetesiTranzakcio(String posId,
            String azonosito, String osszeg, String devizanem, String nyelvkod,
            boolean nevKell, boolean orszagKell, boolean megyeKell,
            boolean telepulesKell, boolean iranyitoszamKell,
            boolean utcaHazszamKell, boolean mailCimKell,
            boolean kozlemenyKell, boolean vevoVisszaigazolasKell,
            boolean ugyfelRegisztracioKell, String regisztraltUgyfelId,
            String shopMegjegyzes, String backURL) throws AxisFault {
    	
    	return fizetesiTranzakcio(posId, azonosito, osszeg, devizanem, nyelvkod,
    			nevKell, orszagKell, megyeKell, 
    			telepulesKell, iranyitoszamKell,
    			utcaHazszamKell, mailCimKell, 
    			kozlemenyKell, vevoVisszaigazolasKell,
    			ugyfelRegisztracioKell, regisztraltUgyfelId, 
    			shopMegjegyzes, backURL, null);
    }

    /**
     * Hromszerepls fizetsi folyamat (WEBSHOPFIZETES) szinkron indtsa
     * Az, hogy a fizets egy- vagy ktlpcss fizets trtnik-e, 
     * explicit mdon kerl megadsra.
     * Digitlis alrs kpzdik.
     *
     * @param posId
     *            webshop azonost
     * @param tranzakcioAzonosito
     *            fizetsi tranzakci azonost
     * @param osszeg
     *            fizetend sszeg, (num, max. 13+2), opcionlis tizedesponttal.
     *            <br />
     *            Nulla is lehet, ha a regisztraltUgyfelId paramter ki van
     *            tltve, s az ugyfelRegisztracioKell rtke igaz. gy kell
     *            ugyanis jelezni azt, hogy nem tnyleges vsrlsi tranzakcit
     *            kell indtani, hanem egy gyfl regisztrlst, vagyis az
     *            gyfl krtyaadatainak bekrst s eltrolst a banki
     *            oldalon.
     * @param devizanem
     *            fizetend devizanem
     * @param nyelvkod
     *            a megjelentend vev oldali fellet nyelve
     * @param nevKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            nevt
     * @param orszagKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "orszg rszt"
     * @param megyeKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "megye rszt"
     * @param telepulesKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "telepls rszt"
     * @param iranyitoszamKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "irnytszm rszt"
     * @param utcaHazszamKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "utca/hzszm rszt"
     * @param mailCimKell
     *            a megjelentend vev oldali felleten be kellkrni a vev
     *            e-mail cmt
     * @param kozlemenyKell
     *            a megjelentend vev oldali felleten fel kell knlni a
     *            kzlemny megadsnak lehetsgt
     * @param vevoVisszaigazolasKell
     *            a tranzakci eredmnyt a vev oldalon meg kell jelenteni
     *            (azaz nem a backURL-re kell irnytani)
     * @param ugyfelRegisztracioKell
     *            ha a regisztraltUgyfelId rtke nem res, akkor megadja, hogy
     *            a megadott azonost jonnan regisztrland-e, vagy mr
     *            regisztrlsra kerlt az OTP Internetes Fizet felletn.
     *            Elbbi esetben a kliens oldali bngszben olyan fizet oldal
     *            fog megjelenni, melyen meg kell adni az azonosthoz tartoz
     *            jelszt, illetve a krtyaadatokat. Utbbi esetben csak az
     *            azonosthoz tartoz jelsz kerl beolvassra az rtestsi
     *            cmen kvl. Ha a regisztraltUgyfelId rtke res, a pamarter
     *            rtke nem kerl felhasznlsra.
     * @param regisztraltUgyfelId
     *            az OTP fizetfelleten regisztrland vagy regisztrlt gyfl
     *            azonost kdja.
     * @param shopMegjegyzes
     *            a webshop megjegyzse a tranzakcihoz a vev rszre
     * @param backURL
     *            a tranzakci vgrehajtsa utn erre az internet cmre kell
     *            irnytani a vev oldalon az gyfelet (ha a
     *            vevoVisszaigazolasKell hamis)
     * @param ketlepcsosFizetes
     * 			  megadja, hogy ktlpcss fizets indtand-e.
     *            True rtk esetn a fizetsi tranzakci ktlpcss lesz, 
     *            azaz a terhelend sszeg csupn zrolsra kerl, 
     *            s gy is marad a bolt ltal indtott lezr tranzakci 
     *            indtsig avagy a zrols elvlsig.
     *
     * @return a tranzakci adatai, mely tartalmazza a webshop ltal megadott
     *         fizetsi adatokat, a vev ltal megadott adatokat, s a
     *         tranzakci eredmnyt a tejests idejvel s a POS autorizcis
     *         kddal egytt.
     */
    public WebShopFizetesAdatokLista fizetesiTranzakcio(String posId,
            String azonosito, String osszeg, String devizanem, String nyelvkod,
            boolean nevKell, boolean orszagKell, boolean megyeKell,
            boolean telepulesKell, boolean iranyitoszamKell,
            boolean utcaHazszamKell, boolean mailCimKell,
            boolean kozlemenyKell, boolean vevoVisszaigazolasKell,
            boolean ugyfelRegisztracioKell, String regisztraltUgyfelId,
            String shopMegjegyzes, String backURL, boolean ketlepcsosFizetes) throws AxisFault {

    	return fizetesiTranzakcio(posId, azonosito, osszeg, devizanem, nyelvkod,
    			nevKell, orszagKell, megyeKell, 
    			telepulesKell, iranyitoszamKell,
    			utcaHazszamKell, mailCimKell, 
    			kozlemenyKell, vevoVisszaigazolasKell,
    			ugyfelRegisztracioKell, regisztraltUgyfelId, 
    			shopMegjegyzes, backURL, Boolean.valueOf(ketlepcsosFizetes));
    }
    
    /**
     * Hromszerepls fizetsi folyamat (WEBSHOPFIZETES) szinkron indtsa.
     * Digitlis alrs kpzdik.
     *
     * @param posId
     *            webshop azonost
     * @param tranzakcioAzonosito
     *            fizetsi tranzakci azonost
     * @param osszeg
     *            fizetend sszeg, (num, max. 13+2), opcionlis tizedesponttal.
     *            <br />
     *            Nulla is lehet, ha a regisztraltUgyfelId paramter ki van
     *            tltve, s az ugyfelRegisztracioKell rtke igaz. gy kell
     *            ugyanis jelezni azt, hogy nem tnyleges vsrlsi tranzakcit
     *            kell indtani, hanem egy gyfl regisztrlst, vagyis az
     *            gyfl krtyaadatainak bekrst s eltrolst a banki
     *            oldalon.
     * @param devizanem
     *            fizetend devizanem
     * @param nyelvkod
     *            a megjelentend vev oldali fellet nyelve
     * @param nevKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            nevt
     * @param orszagKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "orszg rszt"
     * @param megyeKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "megye rszt"
     * @param telepulesKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "telepls rszt"
     * @param iranyitoszamKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "irnytszm rszt"
     * @param utcaHazszamKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "utca/hzszm rszt"
     * @param mailCimKell
     *            a megjelentend vev oldali felleten be kellkrni a vev
     *            e-mail cmt
     * @param kozlemenyKell
     *            a megjelentend vev oldali felleten fel kell knlni a
     *            kzlemny megadsnak lehetsgt
     * @param vevoVisszaigazolasKell
     *            a tranzakci eredmnyt a vev oldalon meg kell jelenteni
     *            (azaz nem a backURL-re kell irnytani)
     * @param ugyfelRegisztracioKell
     *            ha a regisztraltUgyfelId rtke nem res, akkor megadja, hogy
     *            a megadott azonost jonnan regisztrland-e, vagy mr
     *            regisztrlsra kerlt az OTP Internetes Fizet felletn.
     *            Elbbi esetben a kliens oldali bngszben olyan fizet oldal
     *            fog megjelenni, melyen meg kell adni az azonosthoz tartoz
     *            jelszt, illetve a krtyaadatokat. Utbbi esetben csak az
     *            azonosthoz tartoz jelsz kerl beolvassra az rtestsi
     *            cmen kvl. Ha a regisztraltUgyfelId rtke res, a pamarter
     *            rtke nem kerl felhasznlsra.
     * @param regisztraltUgyfelId
     *            az OTP fizetfelleten regisztrland vagy regisztrlt gyfl
     *            azonost kdja.
     * @param shopMegjegyzes
     *            a webshop megjegyzse a tranzakcihoz a vev rszre
     * @param backURL
     *            a tranzakci vgrehajtsa utn erre az internet cmre kell
     *            irnytani a vev oldalon az gyfelet (ha a
     *            vevoVisszaigazolasKell hamis)
     * @param ketlepcsosFizetes
     * 			  megadja, hogy ktlpcss fizets indtand-e.
     *            True rtk esetn a fizetsi tranzakci ktlpcss lesz, 
     *            azaz a terhelend sszeg csupn zrolsra kerl, 
     *            s gy is marad a bolt ltal indtott lezr tranzakci 
     *            indtsig avagy a zrols elvlsig.
     *            Az alaprtelmezett (res) rtk a Bank oldalon rgztett 
     *            alaprtelmezett mdot jelli.       
     *
     * @return a tranzakci adatai, mely tartalmazza a webshop ltal megadott
     *         fizetsi adatokat, a vev ltal megadott adatokat, s a
     *         tranzakci eredmnyt a tejests idejvel s a POS autorizcis
     *         kddal egytt.
     */
    protected WebShopFizetesAdatokLista fizetesiTranzakcio(String posId,
            String azonosito, String osszeg, String devizanem, String nyelvkod,
            boolean nevKell, boolean orszagKell, boolean megyeKell,
            boolean telepulesKell, boolean iranyitoszamKell,
            boolean utcaHazszamKell, boolean mailCimKell,
            boolean kozlemenyKell, boolean vevoVisszaigazolasKell,
            boolean ugyfelRegisztracioKell, String regisztraltUgyfelId,
            String shopMegjegyzes, String backURL,
            Boolean ketlepcsosFizetes) throws AxisFault {

        try {
            WebShopFizetesAdatokLista result;

            category.debug("fizetesiTranzakcio indul...");

            Document dom = WebShopXmlUtils.getRequestSkeleton();

            Node templateNameText = dom
                    .createTextNode(WebShopXmlUtils.TEMPLATENAME_TAGNAME);
            templateNameText.setNodeValue("WEBSHOPFIZETES");

            Node templateName = dom.getDocumentElement().getElementsByTagName(
                    WebShopXmlUtils.TEMPLATENAME_TAGNAME).item(0);
            templateName.appendChild(templateNameText);

            Node variables = dom.getDocumentElement().getElementsByTagName(
                    WebShopXmlUtils.VARIABLES_TAGNAME).item(0);

            /* paramterek beillesztse */
            WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.POSID,
                    posId);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.TRANSACTIONID, azonosito);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.AMOUNT, osszeg);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.EXCHANGE, devizanem);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.LANGUAGECODE, nyelvkod);

            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.NAMENEEDED, nevKell);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.COUNTRYNEEDED, orszagKell);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.COUNTYNEEDED, megyeKell);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.SETTLEMENTNEEDED, telepulesKell);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.ZIPCODENEEDED, iranyitoszamKell);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.STREETNEEDED, utcaHazszamKell);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.MAILADDRESSNEEDED, mailCimKell);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.NARRATIONNEEDED, kozlemenyKell);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.CONSUMERRECEIPTNEEDED,
                    vevoVisszaigazolasKell);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.BACKURL, backURL);

            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.SHOPCOMMENT, shopMegjegyzes);

            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.CONSUMERREGISTRATIONNEEDED,
                    ugyfelRegisztracioKell);
            WebShopXmlUtils
                    .addParameter(dom, variables,
                            WebShopXmlUtils.CONSUMERREGISTRATIONID,
                            regisztraltUgyfelId);

            if (ketlepcsosFizetes != null) {
	            WebShopXmlUtils
	            		.addParameter(dom, variables,
	            				WebShopXmlUtils.TWOSTAGED,
	            				Boolean.TRUE.equals(ketlepcsosFizetes));
            }
            
            // alrs kiszmtsa s paramterknt beszrsa
            byte[] pkcs8PrivateKey = SignatureUtils
                    .loadPrivateKey(getPrivKeyFileName(getWebShopProperties(),
                            posId));
            String[] fields = { posId, azonosito, osszeg, devizanem,
                    regisztraltUgyfelId };
            byte[] autorizaciosKodBytes = SignatureUtils.generateSignature(
                    SignatureUtils.getByteArray(fields), pkcs8PrivateKey);

            String autorizaciosKod = SignatureUtils.toHex(autorizaciosKodBytes,
                    0, (int) autorizaciosKodBytes.length);

            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.CLIENTSIGNATURE, autorizaciosKod);

            // a folyamat indtsa loggozssal
            String domString = WebShopXmlUtils.DOMToXMLString(dom);
            category.info("fizetesiTranzakcio keres:\n" + domString);

            /* Tranzakci adatainak naplzsa egy kln fjlba */
            TransactionLogger transLogger = new TransactionLogger(
                    getTranLogDir(getWebShopProperties(), posId), category);

            transLogger.logHaromszereplosFizetesInditas(posId, azonosito,
                    osszeg, devizanem, nyelvkod, nevKell, orszagKell,
                    megyeKell, telepulesKell, iranyitoszamKell,
                    utcaHazszamKell, mailCimKell, kozlemenyKell,
                    vevoVisszaigazolasKell, ugyfelRegisztracioKell,
                    regisztraltUgyfelId, shopMegjegyzes, backURL,
                    ketlepcsosFizetes);

            WorkflowState state = startWorkflowSynch("WEBSHOPFIZETES",
                    WebShopXmlUtils.DOMToXMLString(dom));

            category.info("fizetesiTranzakcio folyamat azonosio: "
                    + state.getInstanceId());
            System.out.println("Finished: WEBSHOPFIZETES with "
                    + state.getInstanceId() + " at " + new Date());

            WResponse response = getResponse(state.getResult(),
                    "WEBSHOPFIZETES");

            // a folyamat vlasznak loggoza
            domString = WebShopXmlUtils.DOMToXMLString(response
                    .getOriginalResponseDOM().getOwnerDocument());
            category.debug("fizetesiTranzakcio valasz:\n" + domString);

            response.acquire(false, ANSWERFACTORY);

            if (response.isSuccessful()) {
                result = (WebShopFizetesAdatokLista) response.getAnswer()
                        .getData();
                result.setValaszKod("SIKER");
            } else {
                result = new WebShopFizetesAdatokLista();
                result.setValaszKod(getErrorsAsList(response));
            }

            result.setPosId(posId);

            transLogger.logHaromszereplosFizetesBefejezes(azonosito, result);

            category.debug("fizetesiTranzakcio befejezodott.");

            return result;

        } catch (Exception e) {
            // e.printStackTrace();
            category.warn(SERVERERROR_INFO, e);
            throw new AxisFault("", e.getMessage(), null, null);
        }
    }

    /**
     * Ktszerepls fizetsi tranzakci (WEBSHOPFIZETESKETSZEREPLOS) indtsa.
     * Az, hogy egy- vagy ktlpcss fizets trtnik-e, a bolt banki oldal
     * trolt alaprtelmezett belltstl fgg.
     * Digitlis alrs kpzdik.
     *
     * @param posId
     *            tranzakci egyedi azonostja (alfanum, max. 32, azonos a 3
     *            szereplsnl bevezetettel)
     * @param azonosito
     *            a shop azonostja (num, max. 6, azonos a 3 szereplsnl
     *            bevezetettel)
     * @param osszeg
     *            vsrls sszege (num, max. 13+2), opcionlis tizedesponttal
     * @param devizanem
     *            vsrls devizaneme (opcionlis, azonos a 3 szereplsnl
     *            bevezetettel)
     * @param nyelvkod
     *            nyelvkd (azonos a 3 szereplsnl bevezetettel)
     * @param regisztraltUgyfelId
     *            az OTP fizetfelleten regisztrlt gyfl azonost kdja.
     *            Kitltse esetn a kartyaszam, cvc2cvv2, kartyaLejarat adatok
     *            nem kerlnek feldolgozsra, hanem a banki oldalon az adott
     *            azonosthoz eltrolt krtyaadatok kerlnek behelyettestsre
     * @param kartyaszam
     *            krtyaszm (azonos a 3 szereplsnl bevezetettel)
     * @param cvc2cvv2
     *            CVC2/CVV2 kd (azonos a 3 szereplsnl bevezetettel)
     * @param kartyaLejarat
     *            krtya lejrati dtuma, MMyy formban
     * @param vevoNev
     *            vev neve (alfanum, max. 50, opcionlis, csak logozand)
     * @param vevoPostaCim
     *            vev postai cme (alfanum, max. 100, opcionlis)
     * @param vevoIPCim
     *            vev gpnek IP cme (alfanum, max. 15, opcionlis)
     * @param ertesitoMail
     *            vev kirtestsi mailcme (alfanum, max. 50, opcionlis, ha
     *            van, akkor mail kldend a tranzakci eredmnyrl erre a
     *            cmre)
     * @param ertesitoTel
     *            vev kirtestsi telefonszma (alfanum, max. 20, opcionlis,
     *            ha van, akkor SMS kldend a tranzakci eredmnyrl erre a
     *            telefonszmra)
     */
    public WebShopFizetesValasz fizetesiTranzakcioKetszereplos(String posId,
            String azonosito, String osszeg, String devizanem, String nyelvkod,
            String regisztraltUgyfelId, String kartyaszam, String cvc2cvv2,
            String kartyaLejarat, String vevoNev, String vevoPostaCim,
            String vevoIPCim, String ertesitoMail, String ertesitoTel)
            throws AxisFault {

    	return fizetesiTranzakcioKetszereplos(
    			posId, azonosito, osszeg, devizanem, nyelvkod,
    			regisztraltUgyfelId, kartyaszam, cvc2cvv2,
    			kartyaLejarat, vevoNev, vevoPostaCim,
    			vevoIPCim, ertesitoMail, ertesitoTel, null);
    }

    /**
     * Ktszerepls fizetsi tranzakci (WEBSHOPFIZETESKETSZEREPLOS) indtsa.
     * Az, hogy a fizets egy- vagy ktlpcss fizets trtnik-e, 
     * explicit mdon kerl megadsra.
     * Digitlis alrs kpzdik.
     *
     * @param posId
     *            tranzakci egyedi azonostja (alfanum, max. 32, azonos a 3
     *            szereplsnl bevezetettel)
     * @param azonosito
     *            a shop azonostja (num, max. 6, azonos a 3 szereplsnl
     *            bevezetettel)
     * @param osszeg
     *            vsrls sszege (num, max. 13+2), opcionlis tizedesponttal
     * @param devizanem
     *            vsrls devizaneme (opcionlis, azonos a 3 szereplsnl
     *            bevezetettel)
     * @param nyelvkod
     *            nyelvkd (azonos a 3 szereplsnl bevezetettel)
     * @param regisztraltUgyfelId
     *            az OTP fizetfelleten regisztrlt gyfl azonost kdja.
     *            Kitltse esetn a kartyaszam, cvc2cvv2, kartyaLejarat adatok
     *            nem kerlnek feldolgozsra, hanem a banki oldalon az adott
     *            azonosthoz eltrolt krtyaadatok kerlnek behelyettestsre
     * @param kartyaszam
     *            krtyaszm (azonos a 3 szereplsnl bevezetettel)
     * @param cvc2cvv2
     *            CVC2/CVV2 kd (azonos a 3 szereplsnl bevezetettel)
     * @param kartyaLejarat
     *            krtya lejrati dtuma, MMyy formban
     * @param vevoNev
     *            vev neve (alfanum, max. 50, opcionlis, csak logozand)
     * @param vevoPostaCim
     *            vev postai cme (alfanum, max. 100, opcionlis)
     * @param vevoIPCim
     *            vev gpnek IP cme (alfanum, max. 15, opcionlis)
     * @param ertesitoMail
     *            vev kirtestsi mailcme (alfanum, max. 50, opcionlis, ha
     *            van, akkor mail kldend a tranzakci eredmnyrl erre a
     *            cmre)
     * @param ertesitoTel
     *            vev kirtestsi telefonszma (alfanum, max. 20, opcionlis,
     *            ha van, akkor SMS kldend a tranzakci eredmnyrl erre a
     *            telefonszmra)
     * @param ketlepcsosFizetes
     * 			  megadja, hogy ktlpcss fizets indtand-e.
     *            True rtk esetn a fizetsi tranzakci ktlpcss lesz, 
     *            azaz a terhelend sszeg csupn zrolsra kerl, 
     *            s gy is marad a bolt ltal indtott lezr tranzakci 
     *            indtsig avagy a zrols elvlsig.
     *            Az alaprtelmezett (res) rtk a false rtket jelli, teht
     *            a szokvnyos egylpcss azonnali fizets trtnik.	
     */
    public WebShopFizetesValasz fizetesiTranzakcioKetszereplos(String posId,
            String azonosito, String osszeg, String devizanem, String nyelvkod,
            String regisztraltUgyfelId, String kartyaszam, String cvc2cvv2,
            String kartyaLejarat, String vevoNev, String vevoPostaCim,
            String vevoIPCim, String ertesitoMail, String ertesitoTel,
            boolean ketlepcsosFizetes)
            throws AxisFault {

    	return fizetesiTranzakcioKetszereplos(
    			posId, azonosito, osszeg, devizanem, nyelvkod,
    			regisztraltUgyfelId, kartyaszam, cvc2cvv2,
    			kartyaLejarat, vevoNev, vevoPostaCim,
    			vevoIPCim, ertesitoMail, ertesitoTel, Boolean.valueOf(ketlepcsosFizetes));
    }
    
    /**
     * Ktszerepls fizetsi tranzakci (WEBSHOPFIZETESKETSZEREPLOS) indtsa.
     * digitlis alrs kpzdik.
     *
     * @param posId
     *            tranzakci egyedi azonostja (alfanum, max. 32, azonos a 3
     *            szereplsnl bevezetettel)
     * @param azonosito
     *            a shop azonostja (num, max. 6, azonos a 3 szereplsnl
     *            bevezetettel)
     * @param osszeg
     *            vsrls sszege (num, max. 13+2), opcionlis tizedesponttal
     * @param devizanem
     *            vsrls devizaneme (opcionlis, azonos a 3 szereplsnl
     *            bevezetettel)
     * @param nyelvkod
     *            nyelvkd (azonos a 3 szereplsnl bevezetettel)
     * @param regisztraltUgyfelId
     *            az OTP fizetfelleten regisztrlt gyfl azonost kdja.
     *            Kitltse esetn a kartyaszam, cvc2cvv2, kartyaLejarat adatok
     *            nem kerlnek feldolgozsra, hanem a banki oldalon az adott
     *            azonosthoz eltrolt krtyaadatok kerlnek behelyettestsre
     * @param kartyaszam
     *            krtyaszm (azonos a 3 szereplsnl bevezetettel)
     * @param cvc2cvv2
     *            CVC2/CVV2 kd (azonos a 3 szereplsnl bevezetettel)
     * @param kartyaLejarat
     *            krtya lejrati dtuma, MMyy formban
     * @param vevoNev
     *            vev neve (alfanum, max. 50, opcionlis, csak logozand)
     * @param vevoPostaCim
     *            vev postai cme (alfanum, max. 100, opcionlis)
     * @param vevoIPCim
     *            vev gpnek IP cme (alfanum, max. 15, opcionlis)
     * @param ertesitoMail
     *            vev kirtestsi mailcme (alfanum, max. 50, opcionlis, ha
     *            van, akkor mail kldend a tranzakci eredmnyrl erre a
     *            cmre)
     * @param ertesitoTel
     *            vev kirtestsi telefonszma (alfanum, max. 20, opcionlis,
     *            ha van, akkor SMS kldend a tranzakci eredmnyrl erre a
     *            telefonszmra)
     * @param ketlepcsosFizetes
     * 			  megadja, hogy ktlpcss fizets indtand-e.
     *            True rtk esetn a fizetsi tranzakci ktlpcss lesz, 
     *            azaz a terhelend sszeg csupn zrolsra kerl, 
     *            s gy is marad a bolt ltal indtott lezr tranzakci 
     *            indtsig avagy a zrols elvlsig.
     *            Az alaprtelmezett (res) rtk a Bank oldalon rgztett 
     *            alaprtelmezett mdot jelli.       
     */
    protected WebShopFizetesValasz fizetesiTranzakcioKetszereplos(String posId,
            String azonosito, String osszeg, String devizanem, String nyelvkod,
            String regisztraltUgyfelId, String kartyaszam, String cvc2cvv2,
            String kartyaLejarat, String vevoNev, String vevoPostaCim,
            String vevoIPCim, String ertesitoMail, String ertesitoTel,
            Boolean ketlepcsosFizetes)
            throws AxisFault {
        try {
            WebShopFizetesValasz result;
            category.debug("fizetesiTranzakcioKetszereplos indul...");

            Document dom = WebShopXmlUtils.getRequestSkeleton();

            Node templateNameText = dom
                    .createTextNode(WebShopXmlUtils.TEMPLATENAME_TAGNAME);
            templateNameText.setNodeValue("WEBSHOPFIZETESKETSZEREPLOS");

            Node templateName = dom.getDocumentElement().getElementsByTagName(
                    WebShopXmlUtils.TEMPLATENAME_TAGNAME).item(0);
            templateName.appendChild(templateNameText);

            // default rtkek feldolgozsa
            if ((devizanem == null) || (devizanem.trim().length() == 0)) {
                devizanem = DEFAULT_DEVIZANEM;
            }

            Node variables = dom.getDocumentElement().getElementsByTagName(
                    WebShopXmlUtils.VARIABLES_TAGNAME).item(0);

            /* paramterek beillesztse */
            WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.POSID,
                    posId);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.TRANSACTIONID, azonosito);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.AMOUNT, osszeg);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.EXCHANGE, devizanem);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.LANGUAGECODE, nyelvkod);

            WebShopXmlUtils
                    .addParameter(dom, variables,
                            WebShopXmlUtils.CONSUMERREGISTRATIONID,
                            regisztraltUgyfelId);

            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.CARDNUMBER, kartyaszam);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.CVCCVV, cvc2cvv2);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.EXPIRATIONDATE, kartyaLejarat);
            WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.NAME,
                    vevoNev);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.FULLADDRESS, vevoPostaCim);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.IPADDRESS, vevoIPCim);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.MAILADDRESS, ertesitoMail);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.TELEPHONE, ertesitoTel);

            if (ketlepcsosFizetes != null) {
	            WebShopXmlUtils.addParameter(dom, variables,
	    				WebShopXmlUtils.TWOSTAGED, 
	    				Boolean.TRUE.equals(ketlepcsosFizetes));
            }
            
            // alrs kiszmtsa s paramterknt beszrsa
            // ktszerepls esetben tbb mez szerepel az alrsnl...
            byte[] pkcs8PrivateKey = SignatureUtils
                    .loadPrivateKey(getPrivKeyFileName(getWebShopProperties(),
                            posId));
            String[] fields = { posId, azonosito, osszeg, devizanem,
                    kartyaszam, cvc2cvv2, kartyaLejarat, regisztraltUgyfelId };
            byte[] autorizaciosKodBytes = SignatureUtils.generateSignature(
                    SignatureUtils.getByteArray(fields), pkcs8PrivateKey);

            String autorizaciosKod = SignatureUtils.toHex(autorizaciosKodBytes,
                    0, (int) autorizaciosKodBytes.length);

            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.CLIENTSIGNATURE, autorizaciosKod);

            String domString = WebShopXmlUtils.DOMToXMLString(dom);
            category
                    .info("fizetesiTranzakcioKetszereplos keres:\n" + domString);

            /* Tranzakci adatainak naplzsa egy kln fjlba */
            TransactionLogger transLogger = new TransactionLogger(
                    getTranLogDir(getWebShopProperties(), posId), category);

            transLogger.logKetszereplosFizetesInditas(posId, azonosito, osszeg,
                    devizanem, nyelvkod, regisztraltUgyfelId, kartyaszam,
                    cvc2cvv2, kartyaLejarat, vevoNev, vevoPostaCim, vevoIPCim,
                    ertesitoMail, ertesitoTel, ketlepcsosFizetes);

            WorkflowState state = startWorkflowSynch(
                    "WEBSHOPFIZETESKETSZEREPLOS", WebShopXmlUtils
                            .DOMToXMLString(dom));

            category.info("fizetesiTranzakcioKetszereplos folyamat azonosio: "
                    + state.getInstanceId());
            System.out.println("Finished: WEBSHOPFIZETESKETSZEREPLOS with "
                    + state.getInstanceId() + " at " + new Date());

            WResponse response = getResponse(state.getResult(),
                    "WEBSHOPFIZETESKETSZEREPLOS");

            // a folyamat vlasznak loggoza
            domString = WebShopXmlUtils.DOMToXMLString(response
                    .getOriginalResponseDOM().getOwnerDocument());
            category.debug("fizetesiTranzakcioKetszereplos valasz:\n"
                    + domString);

            response.acquire(false, ANSWERFACTORY);

            if (response.isSuccessful()) {
                result = (WebShopFizetesValasz) response.getAnswer().getData();
                result.setValaszKod("SIKER");
            } else {
                result = new WebShopFizetesValasz();
                result.setValaszKod(getErrorsAsList(response));
            }

            result.setPosId(posId);
            result.setAzonosito(azonosito);

            transLogger.logKetszereplosFizetesBefejezes(azonosito, result);

            category.debug("fizetesiTranzakcioKetszereplos befejezodott.");

            return result;

        } catch (Exception e) {
            // e.printStackTrace();
            category.warn(SERVERERROR_INFO, e);
            throw new AxisFault("RENDSZERHIBA", e.getMessage(), null, null);
        }

    }

    /**
     * Hromszerepls fizetsi folyamat (WEBSHOPFIZETES) szinkron indtsa
     * Digitlis alrs kpzdik.
     *
     * @param posId
     *            webshop azonost
     * @param tranzakcioAzonosito
     *            fizetsi tranzakci azonost
     * @param jovahagyo
     * 			  megadja, hogy a lezrs jvhagy vagy tilt jelleg, 
     *            azaz a ktlpcss fizets sorn zrolt sszeg tnylegesen
     *            be kell-e terhelni a vev szmljn, avagy stornzni
     *            kell a ttelt.
     *
     * @return a tranzakci adatai, mely tartalmazza a webshop ltal megadott
     *         fizetsi adatokat, a vev ltal megadott adatokat, s a
     *         tranzakci eredmnyt a tejests idejvel s a POS autorizcis
     *         kddal egytt.
     */
    public WebShopFizetesValasz fizetesiTranzakcioLezaras(
    		String posId,
            String azonosito, 
            boolean jovahagyo) throws AxisFault {

        try {
        	WebShopFizetesValasz result;

            category.debug("fizetesiTranzakcioLezaras indul...");

            Document dom = WebShopXmlUtils.getRequestSkeleton();

            Node templateNameText = dom
                    .createTextNode(WebShopXmlUtils.TEMPLATENAME_TAGNAME);
            templateNameText.setNodeValue(WebShopXmlUtils.WF_KETLEPCSOSFIZETESLEZARAS);

            Node templateName = dom.getDocumentElement().getElementsByTagName(
                    WebShopXmlUtils.TEMPLATENAME_TAGNAME).item(0);
            templateName.appendChild(templateNameText);

            Node variables = dom.getDocumentElement().getElementsByTagName(
                    WebShopXmlUtils.VARIABLES_TAGNAME).item(0);

            /* paramterek beillesztse */
            WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.POSID,
                    posId);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.TRANSACTIONID, azonosito);
            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.APPROVED, jovahagyo);
            
            // alrs kiszmtsa s paramterknt beszrsa
            byte[] pkcs8PrivateKey = SignatureUtils
                    .loadPrivateKey(getPrivKeyFileName(getWebShopProperties(),
                            posId));
            String[] fields = { posId, azonosito };
            byte[] autorizaciosKodBytes = SignatureUtils.generateSignature(
                    SignatureUtils.getByteArray(fields), pkcs8PrivateKey);

            String autorizaciosKod = SignatureUtils.toHex(autorizaciosKodBytes,
                    0, (int) autorizaciosKodBytes.length);

            WebShopXmlUtils.addParameter(dom, variables,
                    WebShopXmlUtils.CLIENTSIGNATURE, autorizaciosKod);

            // a folyamat indtsa loggozssal
            String domString = WebShopXmlUtils.DOMToXMLString(dom);
            category.info("fizetesiTranzakcioLezaras keres:\n" + domString);

            /* Tranzakci adatainak naplzsa egy kln fjlba */
            TransactionLogger transLogger = new TransactionLogger(
                    getTranLogDir(getWebShopProperties(), posId), category);

            transLogger.logFizetesLezarasInditas(posId, azonosito, jovahagyo);

            WorkflowState state = startWorkflowSynch(
            		WebShopXmlUtils.WF_KETLEPCSOSFIZETESLEZARAS,
                    WebShopXmlUtils.DOMToXMLString(dom));

            category.info("fizetesiTranzakcioLezaras folyamat azonosio: "
                    + state.getInstanceId());
            System.out.println(
            		"Finished: " 
            		+ WebShopXmlUtils.WF_KETLEPCSOSFIZETESLEZARAS 
            		+ " with "
                    + state.getInstanceId() + " at " + new Date());

            WResponse response = getResponse(
            		state.getResult(),
            		WebShopXmlUtils.WF_KETLEPCSOSFIZETESLEZARAS);

            // a folyamat vlasznak loggoza
            domString = WebShopXmlUtils.DOMToXMLString(response
                    .getOriginalResponseDOM().getOwnerDocument());
            category.debug("fizetesiTranzakcioLezaras valasz:\n" + domString);

            response.acquire(false, ANSWERFACTORY);

            if (response.isSuccessful()) {
                result = (WebShopFizetesValasz) response.getAnswer()
                        .getData();
                result.setValaszKod("SIKER");
            } else {
                result = new WebShopFizetesValasz();
                result.setValaszKod(getErrorsAsList(response));
            }

            result.setPosId(posId);

            transLogger.logFizetesLezarasBefejezes(azonosito, result);

            category.debug("fizetesiTranzakcioLezaras befejezodott.");

            return result;

        } catch (Exception e) {
            // e.printStackTrace();
            category.warn(SERVERERROR_INFO, e);
            throw new AxisFault("", e.getMessage(), null, null);
        }
    }

}
