<?php

error_reporting(E_ERROR | E_PARSE);

define('WEBSHOP_LIB_VER', '5.0');

if (!defined('WEBSHOP_LIB_DIR')) define('WEBSHOP_LIB_DIR', dirname(dirname(dirname(__FILE__))) );

if (!defined('WEBSHOP_CONF_DIR')) {
	define('WEBSHOP_CONF_DIR', dirname(dirname(dirname(dirname(__FILE__)))) . DIRECTORY_SEPARATOR . 'config');
}

define('WEBSHOPSERVICE_CONFIGURATION', WEBSHOP_CONF_DIR . DIRECTORY_SEPARATOR . 'otp_webshop_client.conf');

define('LOG4PHP_DIR', WEBSHOP_LIB_DIR . '/apache/log4php');
define('LOG4PHP_CONFIGURATION', WEBSHOPSERVICE_CONFIGURATION);

require_once WEBSHOP_LIB_DIR . '/iqsys/otpwebshop/factory/WResponse.php';
require_once WEBSHOP_LIB_DIR . '/iqsys/otpwebshop/util/RequestUtils.php';
require_once WEBSHOP_LIB_DIR . '/iqsys/otpwebshop/util/WebShopXmlUtils.php';
require_once WEBSHOP_LIB_DIR . '/iqsys/otpwebshop/util/SignatureUtils.php';
require_once WEBSHOP_LIB_DIR . '/iqsys/otpwebshop/util/SoapUtils.php';
require_once WEBSHOP_LIB_DIR . '/iqsys/otpwebshop/util/TransactionLogger.php';
require_once WEBSHOP_LIB_DIR . '/iqsys/otpwebshop/util/DefineConst.php';
require_once WEBSHOP_LIB_DIR . '/iqsys/otpwebshop/util/ConfigUtils.php';

require_once LOG4PHP_DIR . '/LoggerManager.php';

/**
 * WebShop szolgltatsok meghvsa a Bank SOAP felletnek kzvetlen elrsvel.
 *
 * A WebShop-ok rszre ksztett PHP eljrs gyjtemny. 
 * 
 * A web-alkalmazst lokkisal, WebShop-onknt kell telepteni.Ezek az
 * alkalmazsok hvjk meg az OTP middleware rendszernek megfelel WebShop
 * folyamatait: 
 * - ping 
 * - tranzakci azonost generls 
 * - hromszerepls fizetsi folyamat indtsa 
 * - ktszerepls fizetsi folyamat indtsa 
 * - tranzakci adatok, tranzakci sttusz lekrdezse
 * - ktlpcss fizetsi tranzakci lezrsa
 * - fizets jvrs indtsa
 *
 * A fenti szolgltatsok kzvetlen mdon, az gynevezett OTP MWAccess felleten
 * is meghvhatak, de ott lnyegesen sszetettebb feladat hrul a WebShop
 * kliens oldali fejesztkre. A kliens oldali WebShop szerver az albbi
 * funkcik vgrehajtsval knnyti a fejlsztst: - nem kell folyamatokat
 * indt xml-eket sszelltani s nem kell vlasz xml- eket rtelmezni.
 * Egyszeren mezszinten kell megadni a bemen adatokat, s ugyancsak
 * mezszinten rhetek el a vlasz adatai. - az alkalmazs automatikusan
 * legenerlja a digitlis alrst azoknl a mveleteknl, ahol ez szksges. -
 * automatikusan naplzsra kerlnek a kommunikci elemei: a kapcsoldsi
 * paramterek, a bejv s kimen SOAP krsek, a folyamat input- s answer
 * xml- jei. A naplzs rszletessge konfigurlhat.
 *
 * @version 4.0
 */

class WebShopService {

    /**
    * Log4php naplz objektum.
    * 
    * @var mixed
    */
    var $logger;
    
    /**
    * Az otp_webshop_client.conf konfigurcis fjl tartalma.
    * 
    * @var array
    */
    var $property;

    /**
    * A bankkal kommunikl SOAP kliens.
    * 
    * $var mixed
    */
    var $soapClient;
    
    /**
    * Az utoljra indtott banki tranzakcihoz tartoz inputXml szveges tartalma
    * 
    * @var string
    */
    var $lastInputXml = NULL;

    /**
    * Az utoljra indtott banki tranzakcihoz tartoz outputXml szveges tartalma
    * 
    * @var string
    */
    var $lastOutputXml = NULL;

    /**
    * A banki tranzakcikhoz tartoz "beszdes nevek", mellyekkel naplzsra kerlnek
    * 
    * @var array
    */
    var $operationLogNames = array(
        'tranzakcioAzonositoGeneralas' => 'tranzakcioAzonositoGeneralas',
        'fizetesiTranzakcioKetszereplos' => 'fizetesiTranzakcioKetszereplos',
        'fizetesiTranzakcio' => 'fizetesiTranzakcio',
        'tranzakcioStatuszLekerdezes' => 'tranzakcioStatuszLekerdezes',
        'ketlepcsosFizetesLezaras' => 'ketlepcsosFizetesLezaras',
        'fizetesJovairas' => 'fizetesJovairas',
        'kulcsLekerdezes' => 'kulcsLekerdezes'
    );
        
    /**
    * Konstruktor.
    * 
    * - inicializldik a log4php
    * - beolvassra kerlt a konfigurcis llomny
    * - pldnyosodik a SOAP kliens
    */
    function WebShopService() {
        $this->logger =& LoggerManager::getLogger("WebShopClient");
        $this->logger->debug("OTPWebShopService (PHP) pldnyosts...");
        
        $this->property = parse_ini_file(WEBSHOPSERVICE_CONFIGURATION);

        $this->logger->debug("OTPMW szerver url: " . ConfigUtils::getConfigParam($this->property, PROPERTY_OTPMWSERVERURL));

        if (ConfigUtils::getConfigParam($this->property, PROPERTY_HTTPSPROXYHOST)) {
            $this->logger->debug("Kliens https proxy host: " . ConfigUtils::getConfigParam($this->property, PROPERTY_HTTPSPROXYHOST));
        }
        if (ConfigUtils::getConfigParam($this->property, PROPERTY_HTTPSPROXYPORT)) {
            $this->logger->debug("Kliens https proxy port: " . ConfigUtils::getConfigParam($this->property, PROPERTY_HTTPSPROXYPORT));
        }
        if (ConfigUtils::getConfigParam($this->property, PROPERTY_HTTPSPROXYUSER)) {
            $this->logger->debug("Kliens https proxy user: " . ConfigUtils::getConfigParam($this->property, PROPERTY_HTTPSPROXYUSER));
        }
        if (ConfigUtils::getConfigParam($this->property, PROPERTY_HTTPSPROXYPASSWORD)) {
            $this->logger->debug("Kliens https proxy password: " . "******");
        }
        
        $this->soapClient = SoapUtils::createSoapClient($this->property);
    }
    
    /**
     * Egy adott fizetsi tranzakcihoz tartoz privt kulcs llomny elrsi
     * tvonalnak belltsa a konfigurcis paramterek alapjn. Ha adott
     * posId (bolt azonost) esetn a konfigurcis llomny tartalmaz
     * otp.webshop.PRIVATE_KEY_posId=[elrsi t] bejegyzst, akkor az eljrs
     * ezt a bejegyzst adja vlaszul. Egybknt a
     * otp.webshop.PRIVATE_KEY_FILE=[elrsi t] bejegyzsben szereplt. Az els
     * mdszerrel lehet multishop-ot kialaktani, vagyis olyan WebShop boolt
     * oldali szervert, amely tbb bolt krst is ki tudja szolglni, s
     * boltonknt (szksgszeren) eltr privt kulcs alapjn trtnik a
     * digitlis alrs.
     *
     * @param properties
     *            A shop-hoz tartoz konfigurcis belltsok (array)
     * @param posId
     *            A tranzakcit indt shop azonostja
     * @return A megadott shop-hoz tartoz privt kulcs elrsi tvonala
     */
    function getPrivKeyFileName($properties, $posId) {
        $privKeyFileName = ConfigUtils::getConfigParam($properties, PROPERTY_PRIVATEKEYFILE, $posId);
        if (!file_exists($privKeyFileName)) {
            $this->logger->fatal("A privt kulcs fjl nem tallhat: " . $privKeyFileName);
        }
        return $privKeyFileName;
    }
    
    /**
     * Egy adott fizetsi tranzakcihoz tartoz tranzakcis napl llomny
     * elrsi tvonalainak belltsa a konfigurcis paramterek alapjn. Ha
     * adott posId (bolt azonost) esetn a konfigurcis llomny tartalmaz
     * otp.webshop.TRANSACTION_LOG_DIR_posId=[elrsi t] bejegyzst, akkor az
     * eljrs ezt a bejegyzst adja vlaszul. Egybknt a
     * otp.webshop.otp.webshop.TRANSACTION_LOG_DIR=[elrsi t] bejegyzsben
     * szereplt elrsi tvonalat. Az els mdszerrel lehet multishop-ot
     * kialaktani, vagyis olyan WebShop bolt oldali szervert, amely tbb bolt
     * krst is ki tudja szolglni, s boltonknt ms knyvtrba trtnik a
     * tranzakcik naplzsa.
     * Ugyanez a kpzsi szably igaz a otp.webshop.transaction_log_dir.SUCCESS_DIR
     * s otp.webshop.transaction_log_dir.FAILED_DIR paramterekre is.
     *
     * @param properties
     *            A shop-hoz tartoz konfigurcis belltsok
     * @param posId
     *            A tranzakcit indt shop azonostja
     * @return A megadott shop-hoz tartoz tranzakcis napl llomnyok
     *         knyvtrainak elrsi tvonala: az alaprtelmezett knyvtr,
     *         a sikeres illetve sikertelen tranzakcik knyvtra
     */
    function getTranLogDir($properties, $posId) {
        $tranLogDir = ConfigUtils::getConfigParam($properties, PROPERTY_TRANSACTIONLOGDIR, $posId);
        $tranLogSuccessDir = ConfigUtils::getConfigParam($properties, PROPERTY_TRANSACTIONLOG_SUCCESS_DIR, $posId);
        $tranLogFailedDir = ConfigUtils::getConfigParam($properties, PROPERTY_TRANSACTIONLOG_FAILED_DIR, $posId);
        return array ($tranLogDir, $tranLogSuccessDir, $tranLogFailedDir);
    }

    /**
     * @desc A banki fellet Ping szolgltatsnak meghvsa. 
     * Mivel tranzakci indts nem trtnik, a sikeres ping
     * esetn sem garantlt az, hogy az egyes fizetsi tranzakcik
     * sikeresen el is indthatk -  csupn az biztos, hogy a
     * hlzati architektrn keresztl sikeresen elrhet a
     * banki fellet. 
     * 
     * Digitlis alrs nem kpzdik.
     * 
     * @return boolean true sikeres ping-ets esetn, egybknt false.
     */
    function ping() {
        $this->logger->debug("ping indul...");
        $result = SoapUtils::ping($this->soapClient, $this->logger);
        $this->logger->debug("ping befejezdtt.");
        return $result;
    }
    
    /**
     * WEBSHOPTRANZAZONGENERALAS folyamat szinkron indtsa. 
     * 
     * @param string $posId 
     *        webshop azonost
     * 
     * @return WResponse a tranzakci vlaszt reprezentl value object.
     *         Sikeres vgrehajts esetn a vlasz adatokat WebShopTranzAzon
     *         objektum reprezentlja.
     *         Kommunikcis hiba esetn a finished flag false rtk lesz!
     */
    function tranzakcioAzonositoGeneralas($posId) {
        $this->logger->debug($this->operationLogNames["tranzakcioAzonositoGeneralas"] . " indul...");
        
        $dom = WebShopXmlUtils::getRequestSkeleton(WF_TRANZAZONGENERALAS, $variables);
        WebShopXmlUtils::addParameter($dom, $variables, CLIENTCODE, CLIENTCODE_VALUE);
        WebShopXmlUtils::addParameter($dom, $variables, POSID, $posId);
        
        $signatureFields = array(0 => $posId);
        $signatureText = SignatureUtils::getSignatureText($signatureFields);

        $pkcs8PrivateKey = SignatureUtils::loadPrivateKey($this->getPrivKeyFileName($this->property, $posId));
        $signature = SignatureUtils::generateSignature($signatureText, $pkcs8PrivateKey, $this->property, $this->logger);
		
		$attrName = null;
		$attrValue = null;
		
		if (version_compare(PHP_VERSION, '5.4.8', '>=')) {
			$attrName = 'algorithm';
			$attrValue = 'SHA512';
		}
		
        WebShopXmlUtils::addParameter($dom, $variables, CLIENTSIGNATURE, $signature, $attrName, $attrValue);
        
        $this->lastInputXml = WebShopXmlUtils::xmlToString($dom);
        $this->logger->debug($this->operationLogNames["tranzakcioAzonositoGeneralas"] . " keres:\n" . WebShopXmlUtils::xmlToString($dom));     
                
        $workflowState = SoapUtils::startWorkflowSynch(WF_TRANZAZONGENERALAS, $this->lastInputXml, $this->soapClient, $this->logger);
        $response = new WResponse(WF_TRANZAZONGENERALAS, $workflowState);

	    $this->logger->info($this->operationLogNames["tranzakcioAzonositoGeneralas"] . " folyamat azonosito: " 
	        . $response->getInstanceId());

	    // a folyamat vlasznak naplzsa
	    if ($response->isFinished()) {
	        $responseDom = $response->getResponseDOM(); 
            $this->lastOutputXml = WebShopXmlUtils::xmlToString($responseDom);                  
            $this->logger->debug($this->operationLogNames["tranzakcioAzonositoGeneralas"] . " valasz:\n" 
	            . trim($this->lastOutputXml));
	    }
	    else {
	        $this->logger->error($this->operationLogNames["tranzakcioAzonositoGeneralas"] . " hiba!");
            $this->logger->error($workflowState);      
	    }

	    $this->logger->debug($this->operationLogNames["tranzakcioAzonositoGeneralas"] . " befejezodott.");

        return $response;
    }

    /**
     * Hromszerepls fizetsi folyamat (WEBSHOPFIZETES) szinkron indtsa.
     *
     * @param string $posId 
     *        webshop azonost
     * @param string $tranzakcioAzonosito 
     *        fizetsi tranzakci azonost
     * @param mixed $osszeg 
     *        Fizetend sszeg, (num, max. 13+2), opcionlis tizedesponttal.
     *        Nulla is lehet, ha a regisztraltUgyfelId paramter ki van
     *        tltve, s az ugyfelRegisztracioKell rtke igaz. gy kell
     *        ugyanis jelezni azt, hogy nem tnyleges vsrlsi tranzakcit
     *        kell indtani, hanem egy gyfl regisztrlst, vagyis az
     *        gyfl krtyaadatainak bekrst s eltrolst a banki
     *        oldalon.
     * @param string $devizanem 
     *            fizetend devizanem
     * @param string $nyelvkod 
     *            a megjelentend vev oldali fellet nyelve
     * @param mixed $nevKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            nevt
     * @param mixed $orszagKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "orszg rszt"
     * @param mixed $megyeKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "megye rszt"
     * @param mixed $telepulesKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "telepls rszt"
     * @param mixed $iranyitoszamKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "irnytszm rszt"
     * @param mixed $utcaHazszamKell
     *            a megjelentend vev oldali felleten be kell krni a vev
     *            cmnek "utca/hzszm rszt"
     * @param mixed $mailCimKell
     *            a megjelentend vev oldali felleten be kellkrni a vev
     *            e-mail cmt
     * @param mixed $kozlemenyKell
     *            a megjelentend vev oldali felleten fel kell knlni a
     *            kzlemny megadsnak lehetsgt
     * @param mixed $vevoVisszaigazolasKell
     *            a tranzakci eredmnyt a vev oldalon meg kell jelenteni
     *            (azaz nem a backURL-re kell irnytani)
     * @param mixed $ugyfelRegisztracioKell
     *            ha a regisztraltUgyfelId rtke nem res, akkor megadja, hogy
     *            a megadott azonost jonnan regisztrland-e, vagy mr
     *            regisztrlsra kerlt az OTP Internetes Fizet felletn.
     *            Elbbi esetben a kliens oldali bngszben olyan fizet oldal
     *            fog megjelenni, melyen meg kell adni az azonosthoz tartoz
     *            jelszt, illetve a krtyaadatokat. Utbbi esetben csak az
     *            azonosthoz tartoz jelsz kerl beolvassra az rtestsi
     *            cmen kvl. Ha a regisztraltUgyfelId rtke res, a pamarter
     *            rtke nem kerl felhasznlsra.
     * @param string $regisztraltUgyfelId
     *            az OTP fizetfelleten regisztrland vagy regisztrlt gyfl
     *            azonost kdja.
     * @param string $shopMegjegyzes
     *            a webshop megjegyzse a tranzakcihoz a vev rszre
     * @param string $backURL
     *            a tranzakci vgrehajtsa utn erre az internet cmre kell
     *            irnytani a vev oldalon az gyfelet (ha a
     *            vevoVisszaigazolasKell hamis)
     * @param string $zsebAzonosito
     * 			  a cafeteria krtya zseb azonostja.       
     * @param mixed $ketlepcsosFizetes
     * 			  megadja, hogy ktlpcss fizets indtand-e.
     *            True rtk esetn a fizetsi tranzakci ktlpcss lesz, 
     *            azaz a terhelend sszeg csupn zrolsra kerl, 
     *            s gy is marad a bolt ltal indtott lezr tranzakci 
     *            indtsig avagy a zrols elvlsig.
     *            Az alaprtelmezett (res) rtk a Bank oldalon rgztett 
     *            alaprtelmezett mdot jelli.       
     *
     * @return WResponse a tranzakci vlaszt reprezentl value object.
     *         Sikeres vgrehajts esetn a vlasz adatokat WebShopFizetesAdatok
     *         objektum reprezentlja.
     *         Kommunikcis hiba esetn a finished flag false rtk lesz!
     */
    function fizetesiTranzakcio(
            $posId,
            $azonosito, 
            $osszeg, 
            $devizanem, 
            $nyelvkod,
            $nevKell, 
            $orszagKell, 
            $megyeKell,
            $telepulesKell, 
            $iranyitoszamKell,
            $utcaHazszamKell, 
            $mailCimKell,
            $kozlemenyKell, 
            $vevoVisszaigazolasKell,
            $ugyfelRegisztracioKell, 
            $regisztraltUgyfelId,
            $shopMegjegyzes, 
            $backURL,
            $zsebAzonosito,
            $ketlepcsosFizetes = NULL) {

        $this->logger->debug($this->operationLogNames["fizetesiTranzakcio"] . " indul...");

        $dom = WebShopXmlUtils::getRequestSkeleton(WF_HAROMSZEREPLOSFIZETESINDITAS, $variables);

        // default rtkek feldolgozsa
        if (is_null($devizanem) || (trim($devizanem) == "")) {
            $devizanem = DEFAULT_DEVIZANEM;
        }

        /* paramterek beillesztse */
        WebShopXmlUtils::addParameter($dom, $variables, CLIENTCODE, CLIENTCODE_VALUE);
        WebShopXmlUtils::addParameter($dom, $variables, POSID, $posId);
        WebShopXmlUtils::addParameter($dom, $variables, TRANSACTIONID, $azonosito);
        WebShopXmlUtils::addParameter($dom, $variables, AMOUNT, $osszeg);
        WebShopXmlUtils::addParameter($dom, $variables, EXCHANGE, $devizanem);
        WebShopXmlUtils::addParameter($dom, $variables, LANGUAGECODE, $nyelvkod);

        WebShopXmlUtils::addParameter($dom, $variables, NAMENEEDED, RequestUtils::booleanToString($nevKell));
        WebShopXmlUtils::addParameter($dom, $variables, COUNTRYNEEDED, RequestUtils::booleanToString($orszagKell));
        WebShopXmlUtils::addParameter($dom, $variables, COUNTYNEEDED, RequestUtils::booleanToString($megyeKell));
        WebShopXmlUtils::addParameter($dom, $variables, SETTLEMENTNEEDED, RequestUtils::booleanToString($telepulesKell));
        WebShopXmlUtils::addParameter($dom, $variables, ZIPCODENEEDED, RequestUtils::booleanToString($iranyitoszamKell));
        WebShopXmlUtils::addParameter($dom, $variables, STREETNEEDED, RequestUtils::booleanToString($utcaHazszamKell));
        WebShopXmlUtils::addParameter($dom, $variables, MAILADDRESSNEEDED, RequestUtils::booleanToString($mailCimKell));
        WebShopXmlUtils::addParameter($dom, $variables, NARRATIONNEEDED, RequestUtils::booleanToString($kozlemenyKell));
        WebShopXmlUtils::addParameter($dom, $variables, CONSUMERRECEIPTNEEDED, RequestUtils::booleanToString($vevoVisszaigazolasKell));

        WebShopXmlUtils::addParameter($dom, $variables, BACKURL, $backURL);

        WebShopXmlUtils::addParameter($dom, $variables, SHOPCOMMENT, $shopMegjegyzes);

        WebShopXmlUtils::addParameter($dom, $variables, CONSUMERREGISTRATIONNEEDED, $ugyfelRegisztracioKell);
        WebShopXmlUtils::addParameter($dom, $variables, CONSUMERREGISTRATIONID, $regisztraltUgyfelId);

        WebShopXmlUtils::addParameter($dom, $variables, TWOSTAGED, RequestUtils::booleanToString($ketlepcsosFizetes, NULL));
        WebShopXmlUtils::addParameter($dom, $variables, CARDPOCKETID, $zsebAzonosito);

        /* alrs kiszmtsa s paramterknt beszrsa */
        $signatureFields = array(0 => 
            $posId, $azonosito, $osszeg, $devizanem, $regisztraltUgyfelId);
        $signatureText = SignatureUtils::getSignatureText($signatureFields);

        $pkcs8PrivateKey = SignatureUtils::loadPrivateKey($this->getPrivKeyFileName($this->property, $posId));
        $signature = SignatureUtils::generateSignature($signatureText, $pkcs8PrivateKey, $this->property, $this->logger);

        $attrName = null;
		$attrValue = null;
		
		if (version_compare(PHP_VERSION, '5.4.8', '>=')) {
			$attrName = 'algorithm';
			$attrValue = 'SHA512';
		}
		
        WebShopXmlUtils::addParameter($dom, $variables, CLIENTSIGNATURE, $signature, $attrName, $attrValue);

        $this->lastInputXml = WebShopXmlUtils::xmlToString($dom);
        $this->logger->info($this->operationLogNames["fizetesiTranzakcio"] . " keres: " . $posId . " / " . $azonosito);
        $this->logger->debug($this->operationLogNames["fizetesiTranzakcio"] . " keres:\n" . $this->lastInputXml);

        /* Tranzakci adatainak naplzsa egy kln fjlba */
        $transLogger = new TransactionLogger(
                    $this->getTranLogDir($this->property, $posId), $this->logger);

        $transLogger->logHaromszereplosFizetesInditas($posId, $azonosito, 
                $osszeg, $devizanem, $nyelvkod, $nevKell, $orszagKell,
                $megyeKell, $telepulesKell, $iranyitoszamKell,
                $utcaHazszamKell, $mailCimKell, $kozlemenyKell,
                $vevoVisszaigazolasKell, $ugyfelRegisztracioKell,
                $regisztraltUgyfelId, $shopMegjegyzes, $backURL,
                $zsebAzonosito, $ketlepcsosFizetes);
        
        /* A tranzakci indtsa */
        $startTime = time();
        $workflowState = SoapUtils::startWorkflowSynch(WF_HAROMSZEREPLOSFIZETESINDITAS, $this->lastInputXml, $this->soapClient, $this->logger);
        
        if (!is_null($workflowState)) {
            $response = new WResponse(WF_HAROMSZEREPLOSFIZETES, $workflowState);
        }
        else {
            $this->logger->warn($this->operationLogNames["fizetesiTranzakcio"] . " folyamat megszakadt: " 
                . $azonosito . ", pollozs indul..."); 
            // A tranzakci megszakadt, a banki fellet vlaszt nem
            // tudta a kliens fogadni
            $poll = true;
            $resendDelay = 20;
            do {
                $tranzAdatok = $this->tranzakcioPoll($posId, $azonosito, $startTime);
                if ($tranzAdatok === false) {
                    // nem sikerlt a lekrdezs, jraprblkozunk
                    $poll = true;
                    $this->logger->error($this->operationLogNames["fizetesiTranzakcio"] . " poll hiba, azonosito: " . $azonosito);
                }
                else {
                    if ($tranzAdatok->isFizetesFeldolgozasAlatt()) {
                        // a tranzakci feldolgozs alatt van
                        // mindenkpp rdemes kicsit vrni, s jra pollozni
                    }
                    else {
                        // a tranzakci feldolgozsa befejezdtt 
                        // (lehet sikeres vagy sikertelen az eredmny)
                        $poll = false;
                        $response = new WResponse(WF_HAROMSZEREPLOSFIZETES, null);
                        $this->logger->info($this->operationLogNames["fizetesiTranzakcio"] . " poll befejezve: " 
                            . $azonosito); 
                        // a folyamat vlasznak naplzsa
                        $response->loadAnswerModel($tranzAdatok, $tranzAdatok->isSuccessful(), $tranzAdatok->getPosValaszkod());  
                        $transLogger->logHaromszereplosFizetesBefejezes($azonosito, $posId, $response);
                        return $response;
                    }
                }
                $retryCount++;
                sleep($resendDelay);
            } while ($poll && ($startTime + 660 > time()));
            // pollozunk, amg van rtelme, de legfeljebb 11 percig! 
            
            $this->logger->info($this->operationLogNames["fizetesiTranzakcio"] 
                . $azonosito . ", pollozs befejezve..."); 
        }
  
        // a folyamat vlasznak naplzsa
        if ($response->isFinished()) {
            $this->logger->info($this->operationLogNames["fizetesiTranzakcio"] . " folyamat azonosito: " 
                . $response->getInstanceId());
            $responseDom = $response->getResponseDOM();
            $this->lastOutputXml = WebShopXmlUtils::xmlToString($responseDom);
            $this->logger->debug($this->operationLogNames["fizetesiTranzakcio"] . " valasz:\n" 
                . trim($this->lastOutputXml));
            $transLogger->logHaromszereplosFizetesBefejezes($azonosito, $posId, $response);
        }
        else {
            $this->logger->error($this->operationLogNames["fizetesiTranzakcio"] . " hiba, azonosito: " . $azonosito);
            $this->logger->error($workflowState);
        }
                
        $this->logger->debug($this->operationLogNames["fizetesiTranzakcio"] . " befejezodott.");

        return $response;
    }
    
    /**
     * WEBSHOPTRANZAKCIOLEKERDEZES folyamat szinkron indtsa.
     * 
     * @param string $posId webshop azonost
     * @param string $azonosito lekrdezend tranzakci azonost
     * @param mixed $maxRekordSzam maximlis rekordszm (int / string)
     * @param mixed $idoszakEleje lekrdezend idszak eleje 
     *        .HH.NN :PP:MM alak string rtk vagy int timestamp
     * @param mixed $idoszakEleje lekrdezend idszak vge
     *        .HH.NN :PP:MM alak string rtk vagy int timestamp
     * 
     * @return WResponse a tranzakci vlaszt reprezentl value object.
     *         Sikeres vgrehajts esetn a vlasz adatokat WebShopAdatokLista
     *         objektum reprezentlja.
     *         Kommunikcis hiba esetn a finished flag false rtk lesz!
     */
    function tranzakcioStatuszLekerdezes(
            $posId,
            $azonosito, 
            $maxRekordSzam, 
            $idoszakEleje,
            $idoszakVege) {
                
        $this->logger->debug($this->operationLogNames["tranzakcioStatuszLekerdezes"] . " indul...");

        $dom = WebShopXmlUtils::getRequestSkeleton(WF_TRANZAKCIOSTATUSZ, $variables);

        $idoszakEleje = RequestUtils::dateToString($idoszakEleje);
        $idoszakVege = RequestUtils::dateToString($idoszakVege);
        
        /* paramterek beillesztse */
        WebShopXmlUtils::addParameter($dom, $variables, CLIENTCODE, CLIENTCODE_VALUE);
        WebShopXmlUtils::addParameter($dom, $variables, POSID, $posId);
        WebShopXmlUtils::addParameter($dom, $variables, TRANSACTIONID, $azonosito);
        WebShopXmlUtils::addParameter($dom, $variables, QUERYMAXRECORDS, $maxRekordSzam);
        WebShopXmlUtils::addParameter($dom, $variables, QUERYSTARTDATE, $idoszakEleje);
        WebShopXmlUtils::addParameter($dom, $variables, QUERYENDDATE, $idoszakVege);

        /* alrs kiszmtsa s paramterknt beszrsa */
        $signatureFields = array(0 => 
            $posId, $azonosito, 
            $maxRekordSzam, $idoszakEleje, $idoszakVege );
        $signatureText = SignatureUtils::getSignatureText($signatureFields);

        $pkcs8PrivateKey = SignatureUtils::loadPrivateKey($this->getPrivKeyFileName($this->property, $posId));
        $signature = SignatureUtils::generateSignature($signatureText, $pkcs8PrivateKey, $this->property, $this->logger);

        $attrName = null;
		$attrValue = null;
		
		if (version_compare(PHP_VERSION, '5.4.8', '>=')) {
			$attrName = 'algorithm';
			$attrValue = 'SHA512';
		}
		
        WebShopXmlUtils::addParameter($dom, $variables, CLIENTSIGNATURE, $signature, $attrName, $attrValue);

        $this->lastInputXml = WebShopXmlUtils::xmlToString($dom);
        $this->logger->debug($this->operationLogNames["tranzakcioStatuszLekerdezes"] . " keres:\n" . $this->lastInputXml);

        /* a folyamat indtsa */
        $workflowState = SoapUtils::startWorkflowSynch(WF_TRANZAKCIOSTATUSZ, $this->lastInputXml, $this->soapClient, $this->logger);
        $response = new WResponse(WF_TRANZAKCIOSTATUSZ, $workflowState);

        /* a folyamat vlasznak naplzsa */
        if ($response->isFinished()) {
            $this->logger->info($this->operationLogNames["tranzakcioStatuszLekerdezes"] . " folyamat azonosito: " 
                . $response->getInstanceId());
            $responseDom = $response->getResponseDOM();
            $this->lastOutputXml = WebShopXmlUtils::xmlToString($responseDom);
            $this->logger->debug($this->operationLogNames["tranzakcioStatuszLekerdezes"] . " valasz:\n" 
                . trim($this->lastOutputXml));
        }
        else {
            $this->logger->error($this->operationLogNames["tranzakcioStatuszLekerdezes"] . " hiba!");
            $this->logger->error($workflowState);            
        }

        $this->logger->debug($this->operationLogNames["tranzakcioStatuszLekerdezes"] . " befejezodott.");

        return $response;
    }
    
    /**
     * Ktszerepls fizetsi tranzakci (WEBSHOPFIZETESKETSZEREPLOS) indtsa.
     *
     * @param string $posId
     *            tranzakci egyedi azonostja (alfanum, max. 32, azonos a 3
     *            szereplsnl bevezetettel)
     * @param string $azonosito
     *            a shop azonostja (num, max. 6, azonos a 3 szereplsnl
     *            bevezetettel)
     * @param mixed $osszeg
     *            vsrls sszege (num, max. 13+2), opcionlis tizedesponttal
     * @param string $devizanem
     *            vsrls devizaneme (opcionlis, azonos a 3 szereplsnl
     *            bevezetettel)
     * @param stirng $nyelvkod
     *            nyelvkd (azonos a 3 szereplsnl bevezetettel)
     * @param string $regisztraltUgyfelId
     *            az OTP fizetfelleten regisztrlt gyfl azonost kdja.
     *            Kitltse esetn a kartyaszam, cvc2cvv2, kartyaLejarat adatok
     *            nem kerlnek feldolgozsra, hanem a banki oldalon az adott
     *            azonosthoz eltrolt krtyaadatok kerlnek behelyettestsre
     * @param string $kartyaszam
     *            krtyaszm (azonos a 3 szereplsnl bevezetettel)
     * @param string $cvc2cvv2
     *            CVC2/CVV2 kd (azonos a 3 szereplsnl bevezetettel)
     * @param string $kartyaLejarat
     *            krtya lejrati dtuma, MMyy formban
     * @param string $vevoNev
     *            vev neve (alfanum, max. 50, opcionlis, csak logozand)
     * @param string $vevoPostaCim
     *            vev postai cme (alfanum, max. 100, opcionlis)
     * @param string $vevoIPCim
     *            vev gpnek IP cme (alfanum, max. 15, opcionlis)
     * @param string $ertesitoMail
     *            vev kirtestsi mailcme (alfanum, max. 50, opcionlis, ha
     *            van, akkor mail kldend a tranzakci eredmnyrl erre a
     *            cmre)
     * @param string $ertesitoTel
     *            vev kirtestsi telefonszma (alfanum, max. 20, opcionlis,
     *            ha van, akkor SMS kldend a tranzakci eredmnyrl erre a
     *            telefonszmra)
     * @param mixed $ketlepcsosFizetes
     * 			  megadja, hogy ktlpcss fizets indtand-e.
     *            True rtk esetn a fizetsi tranzakci ktlpcss lesz, 
     *            azaz a terhelend sszeg csupn zrolsra kerl, 
     *            s gy is marad a bolt ltal indtott lezr tranzakci 
     *            indtsig avagy a zrols elvlsig.
     *            Az alaprtelmezett (res) rtk a Bank oldalon rgztett 
     *            alaprtelmezett mdot jelli.       
     * @param string $zsebAzonosito
     * 			  a cafeteria krtya zseb azonostja.     
     * @return WResponse a tranzakci vlaszt reprezentl value object.
     *         Sikeres vgrehajts esetn a vlasz adatokat WebShopFizetesValasz
     *         objektum reprezentlja.
     *         Kommunikcis hiba esetn a finished flag false rtk lesz!
     */
    function fizetesiTranzakcioKetszereplos(
            $posId,
            $azonosito, 
            $osszeg, 
            $devizanem, 
            $nyelvkod,
            $regisztraltUgyfelId, 
            $kartyaszam, 
            $cvc2cvv2,
            $kartyaLejarat, 
            $vevoNev, 
            $vevoPostaCim,
            $vevoIPCim, 
            $ertesitoMail, 
            $ertesitoTel,
            $ketlepcsosFizetes = NULL,
            $zsebAzonosito) {

        $this->logger->debug($this->operationLogNames["fizetesiTranzakcioKetszereplos"] . " indul...");

        $dom = WebShopXmlUtils::getRequestSkeleton(WF_KETSZEREPLOSFIZETES, $variables);

        // default rtkek feldolgozsa
        if (is_null($devizanem) || (trim($devizanem) == "")) {
            $devizanem = DEFAULT_DEVIZANEM;
        }

        /* paramterek beillesztse */
        WebShopXmlUtils::addParameter($dom, $variables, CLIENTCODE, CLIENTCODE_VALUE);
        WebShopXmlUtils::addParameter($dom, $variables, POSID, $posId);
        WebShopXmlUtils::addParameter($dom, $variables, TRANSACTIONID, $azonosito);
        WebShopXmlUtils::addParameter($dom, $variables, AMOUNT, $osszeg);
        WebShopXmlUtils::addParameter($dom, $variables, EXCHANGE, $devizanem);
        WebShopXmlUtils::addParameter($dom, $variables, LANGUAGECODE, $nyelvkod);
        WebShopXmlUtils::addParameter($dom, $variables, CONSUMERREGISTRATIONID, $regisztraltUgyfelId);
        WebShopXmlUtils::addParameter($dom, $variables, CARDNUMBER, $kartyaszam);
        WebShopXmlUtils::addParameter($dom, $variables, CVCCVV, $cvc2cvv2);
        WebShopXmlUtils::addParameter($dom, $variables, EXPIRATIONDATE, $kartyaLejarat);
        WebShopXmlUtils::addParameter($dom, $variables, NAME, $vevoNev);
        WebShopXmlUtils::addParameter($dom, $variables, FULLADDRESS, $vevoPostaCim);
        WebShopXmlUtils::addParameter($dom, $variables, IPADDRESS, $vevoIPCim);
        WebShopXmlUtils::addParameter($dom, $variables, MAILADDRESS, $ertesitoMail);
        WebShopXmlUtils::addParameter($dom, $variables, TELEPHONE, $ertesitoTel);
        WebShopXmlUtils::addParameter($dom, $variables, TWOSTAGED, RequestUtils::booleanToString($ketlepcsosFizetes));
		WebShopXmlUtils::addParameter($dom, $variables, CARDPOCKETID, $zsebAzonosito);
		
        /* alrs kiszmtsa s paramterknt beszrsa */
        $signatureFields = array(0 => 
            $posId, $azonosito, $osszeg, $devizanem,
            $kartyaszam, $cvc2cvv2, $kartyaLejarat, $regisztraltUgyfelId);
        $signatureText = SignatureUtils::getSignatureText($signatureFields);

        $pkcs8PrivateKey = SignatureUtils::loadPrivateKey($this->getPrivKeyFileName($this->property, $posId));
        $signature = SignatureUtils::generateSignature($signatureText, $pkcs8PrivateKey, $this->property, $this->logger);

        $attrName = null;
		$attrValue = null;
		
		if (version_compare(PHP_VERSION, '5.4.8', '>=')) {
			$attrName = 'algorithm';
			$attrValue = 'SHA512';
		}
		
        WebShopXmlUtils::addParameter($dom, $variables, CLIENTSIGNATURE, $signature, $attrName, $attrValue);

        $this->lastInputXml = WebShopXmlUtils::xmlToString($dom);
        $this->logger->info($this->operationLogNames["fizetesiTranzakcioKetszereplos"] . " keres: " . $posId . " / " . $azonosito);
        $this->logger->debug($this->operationLogNames["fizetesiTranzakcioKetszereplos"] . " keres:\n" . $this->removeSensitiveParams($this->lastInputXml));

        /* Tranzakci adatainak naplzsa egy kln fjlba */
        $transLogger = new TransactionLogger(
                    $this->getTranLogDir($this->property, $posId), $this->logger);

        $transLogger->logKetszereplosFizetesInditas($posId, $azonosito, $osszeg,
                    $devizanem, $nyelvkod, $regisztraltUgyfelId, $kartyaszam,
                    $cvc2cvv2, $kartyaLejarat, $vevoNev, $vevoPostaCim, $vevoIPCim,
                    $ertesitoMail, $ertesitoTel, $ketlepcsosFizetes, $zsebAzonosito);

        /* Tranzakci indtsa */
        $workflowState = SoapUtils::startWorkflowSynch(WF_KETSZEREPLOSFIZETES, $this->lastInputXml, $this->soapClient, $this->logger);
        $response = new WResponse(WF_KETSZEREPLOSFIZETES, $workflowState);

        /* a folyamat vlasznak naplzsa */
        if ($response->isFinished()) {
            $this->logger->info($this->operationLogNames["fizetesiTranzakcioKetszereplos"] . " folyamat azonosito: " 
                . $response->getInstanceId());
            $responseDom = $response->getResponseDOM();
            $this->lastOutputXml = WebShopXmlUtils::xmlToString($responseDom);
 	        $this->logger->debug($this->operationLogNames["fizetesiTranzakcioKetszereplos"] . " valasz:\n" 
                . trim($this->lastOutputXml));
            $transLogger->logKetszereplosFizetesBefejezes($azonosito, $posId, $response);
        }
        else {
            $this->logger->error($this->operationLogNames["fizetesiTranzakcioKetszereplos"] . " hiba, azonosito: " . $azonosito);  
            $this->logger->error($workflowState);
        }

        $this->logger->debug($this->operationLogNames["fizetesiTranzakcioKetszereplos"] . " befejezodott.");

        return $response;
    }
  
    /**
     * Ktlpcss fizetsi tranzakci lezrsnak (WEBSHOPFIZETESLEZARAS) indtsa.
     *
     * @param string $posId
     *            a shop azonostja 
     * @param string $azonosito
     *            a lezrand fizetsi tranzakci egyedi azonostja 
     * @param mixed $jovahagyo
     * 			  megadja, hogy a lezrs jvhagy vagy tilt jelleg, 
     *            azaz a ktlpcss fizets sorn zrolt sszeg tnylegesen
     *            be kell-e terhelni a vev szmljn, avagy stornzni
     *            kell a ttelt.
	 * @param mixed $osszeg
     *            kisebb sszeggel trtn komplettrozs esetn megadhat sszeg
	 *            (num, max. 13+2), opcionlis tizedesponttal
     * 
     * @return WResponse a tranzakci vlaszt reprezentl value object.
     *         Sikeres vgrehajts esetn a vlasz adatokat WebShopFizetesValasz
     *         objektum reprezentlja.
     *         Kommunikcis hiba esetn a finished flag false rtk lesz!
     */
    function fizetesiTranzakcioLezaras(
            $posId,
            $azonosito, 
            $jovahagyo,
            $osszeg) {

        $this->logger->debug($this->operationLogNames["ketlepcsosFizetesLezaras"] . " indul...");

        $dom = WebShopXmlUtils::getRequestSkeleton(WF_KETLEPCSOSFIZETESLEZARAS, $variables);

        /* paramterek beillesztse */
        WebShopXmlUtils::addParameter($dom, $variables, CLIENTCODE, CLIENTCODE_VALUE);
        WebShopXmlUtils::addParameter($dom, $variables, POSID, $posId);
        WebShopXmlUtils::addParameter($dom, $variables, TRANSACTIONID, $azonosito);
        WebShopXmlUtils::addParameter($dom, $variables, APPROVED, RequestUtils::booleanToString($jovahagyo));
        WebShopXmlUtils::addParameter($dom, $variables, AMOUNT, $osszeg);
        
        /* alrs kiszmtsa s paramterknt beszrsa */
        $signatureFields = array(0 => $posId, $azonosito);
        $signatureText = SignatureUtils::getSignatureText($signatureFields);

        $pkcs8PrivateKey = SignatureUtils::loadPrivateKey($this->getPrivKeyFileName($this->property, $posId));
        $signature = SignatureUtils::generateSignature($signatureText, $pkcs8PrivateKey, $this->property, $this->logger);

        $attrName = null;
		$attrValue = null;
		
		if (version_compare(PHP_VERSION, '5.4.8', '>=')) {
			$attrName = 'algorithm';
			$attrValue = 'SHA512';
		}
		
        WebShopXmlUtils::addParameter($dom, $variables, CLIENTSIGNATURE, $signature, $attrName, $attrValue);

        $this->lastInputXml = WebShopXmlUtils::xmlToString($dom);
        $this->logger->debug($this->operationLogNames["ketlepcsosFizetesLezaras"] . " keres:\n" . $this->lastInputXml);

        /* Tranzakci adatainak naplzsa egy kln fjlba */
        $transLogger = new TransactionLogger(
                    $this->getTranLogDir($this->property, $posId), $this->logger);

        $transLogger->logFizetesLezarasInditas($posId, $azonosito, $jovahagyo, $osszeg);

        /* Tranzakci indtsa */
        $workflowState = SoapUtils::startWorkflowSynch(WF_KETLEPCSOSFIZETESLEZARAS, $this->lastInputXml, $this->soapClient, $this->logger);
        $response = new WResponse(WF_KETLEPCSOSFIZETESLEZARAS, $workflowState);

        /* a folyamat vlasznak naplzsa */
        if ($response->isFinished()) {
            $this->logger->info($this->operationLogNames["ketlepcsosFizetesLezaras"] . " folyamat azonosito: " 
                . $response->getInstanceId());
 	        $this->lastOutputXml = WebShopXmlUtils::xmlToString($response->getResponseDOM());
            $this->logger->debug($this->operationLogNames["fizetesiTranzakcioLezaras"] . " valasz:\n" 
                . trim($this->lastOutputXml));
            $transLogger->logKetszereplosFizetesBefejezes($azonosito, $posId, $response);
        }
        else {
            $this->logger->error($this->operationLogNames["ketlepcsosFizetesLezaras"] . " hiba, azonosito: " . $azonosito);  
            $this->logger->error($workflowState);
        }

        $this->logger->debug($this->operationLogNames["ketlepcsosFizetesLezaras"] . " befejezodott.");

        return $response;
    }

    /**
     * WEBSHOPTRANZAKCIOLEKERDEZES folyamat szinkron indtsa pollozs cljbl.
     * A bank nem javasolja, hogy pollozsos technikval trtnjen a fizetsi
     * tranzakcik eredmnynek lekrdezse - mindazonltal kommunikcis vagy
     * egyb hiba esetn ez az egyetlen mdja annak, hogy a tranzakci vlaszt
     * utlag le lehessen krdezni.
     * 
     * @param string $posId webshop azonost
     * @param string $azonosito lekrdezend tranzakci azonost
     * @param int $inditas a tranzakci indtsa az indt kliens rja szerint 
     *                     (a lekrdezs +-24 rra fog korltozdni)
     * 
     * @return mixed Sikeres lekrdezs s ltez tranzakci esetn 
     *               a vonatkoz WebShopFizetesAdatok. A tranzakci llapott
     *               ez az objektum fogja tartalmazni - ami utalhat pldul 
     *               vev oldali input vrakozsra vagy feldolgozott sttuszra.
     *               FALSE hibs lekrdezs esetn. (Pl. nem ltezik tranzakci)
     */
    function tranzakcioPoll($posId, $azonosito,  $inditas) {

        $maxRekordSzam = "1";
        $idoszakEleje = $inditas - 60*60*24;
        $idoszakVege = $inditas + 60*60*24;
                
        $tranzAdatok = false;                        
        $response = $this->tranzakcioStatuszLekerdezes($posId, $azonosito, $maxRekordSzam, $idoszakEleje, $idoszakVege);
        if ($response) {
            $answer = $response->getAnswer();
            if ($response->isSuccessful()
                    && $response->getAnswer()
                    && count($answer->getWebShopFizetesAdatok()) > 0) {

                // Sikerlt lekrdezni az adott tranzakci adatt
                $fizetesAdatok = $answer->getWebShopFizetesAdatok();
                $tranzAdatok = reset($fizetesAdatok);
            }
        }
        return $tranzAdatok;
    }
    
    /**
     * Fizets jvrs tranzakci (WEBSHOPFIZETESJOVAIRAS) indtsa.
     *
     * @param string $posId
     *            a shop azonostja 
     * @param string $azonosito
     *            a jvrand fizetsi tranzakci egyedi azonostja 
     * @param mixed $osszeg
     *            vsrls sszege (num, max. 13+2), opcionlis tizedesponttal
     * 
     * @return WResponse a tranzakci vlaszt reprezentl value object.
     *         Sikeres vgrehajts esetn a vlasz adatokat WebShopJovairasValasz
     *         objektum reprezentlja.
     *         Kommunikcis hiba esetn a finished flag false rtk lesz!
     */
    function fizetesJovairas(
            $posId,
            $azonosito, 
            $osszeg) {

        $this->logger->debug($this->operationLogNames["fizetesJovairas"] . " indul...");

        $dom = WebShopXmlUtils::getRequestSkeleton(WF_FIZETESJOVAIRAS, $variables);

        /* paramterek beillesztse */
        WebShopXmlUtils::addParameter($dom, $variables, CLIENTCODE, CLIENTCODE_VALUE);
        WebShopXmlUtils::addParameter($dom, $variables, POSID, $posId);
        WebShopXmlUtils::addParameter($dom, $variables, TRANSACTIONID, $azonosito);
        WebShopXmlUtils::addParameter($dom, $variables, AMOUNT, $osszeg);
        
        /* alrs kiszmtsa s paramterknt beszrsa */
        $signatureFields = array(0 => $posId, $azonosito, $osszeg);
        $signatureText = SignatureUtils::getSignatureText($signatureFields);

        $pkcs8PrivateKey = SignatureUtils::loadPrivateKey($this->getPrivKeyFileName($this->property, $posId));
        $signature = SignatureUtils::generateSignature($signatureText, $pkcs8PrivateKey, $this->property, $this->logger);

        $attrName = null;
		$attrValue = null;
		
		if (version_compare(PHP_VERSION, '5.4.8', '>=')) {
			$attrName = 'algorithm';
			$attrValue = 'SHA512';
		}
		
        WebShopXmlUtils::addParameter($dom, $variables, CLIENTSIGNATURE, $signature, $attrName, $attrValue);

        $this->lastInputXml = WebShopXmlUtils::xmlToString($dom);
        $this->logger->debug($this->operationLogNames["fizetesJovairas"] . " keres:\n" . $this->lastInputXml);

        /* Tranzakci adatainak naplzsa egy kln fjlba */
        $transLogger = new TransactionLogger(
                    $this->getTranLogDir($this->property, $posId), $this->logger);

        $transLogger->logFizetesJovairasInditas($posId, $azonosito, $osszeg);

        /* Tranzakci indtsa */
        $workflowState = SoapUtils::startWorkflowSynch(WF_FIZETESJOVAIRAS, $this->lastInputXml, $this->soapClient, $this->logger);
        $response = new WResponse(WF_FIZETESJOVAIRAS, $workflowState);

        /* a folyamat vlasznak naplzsa */
        if ($response->isFinished()) {
            $this->logger->info($this->operationLogNames["fizetesJovairas"] . " folyamat azonosito: " 
                . $response->getInstanceId());
 	        $this->lastOutputXml = WebShopXmlUtils::xmlToString($response->getResponseDOM());
            $this->logger->debug($this->operationLogNames["fizetesJovairas"] . " valasz:\n" 
                . trim($this->lastOutputXml));
            $transLogger->logKetszereplosFizetesBefejezes($azonosito, $posId, $response);
        }
        else {
            $this->logger->error($this->operationLogNames["fizetesJovairas"] . " hiba, azonosito: " . $azonosito);  
            $this->logger->error($workflowState);
        }

        $this->logger->debug($this->operationLogNames["fizetesJovairas"] . " befejezodott.");

        return $response;
    }
    
    
    /**
     * WEBSHOPTRANZAKCIOLEKERDEZES folyamat szinkron indtsa.
     * 
     * @param string $posId webshop azonost
     * @param string $muvelet mvelet
     * 
     * @return WResponse a tranzakci vlaszt reprezentl value object.
     *         Sikeres vgrehajts esetn a vlasz adatokat WebShopAdatokLista
     *         objektum reprezentlja.
     *         Kommunikcis hiba esetn a finished flag false rtk lesz!
     */
    function kulcsLekerdezes($posId, $muvelet) {
        
        global $variables;
        
        $this->logger->debug($this->operationLogNames["kulcsLekerdezes"] . " indul...");

        $dom = WebShopXmlUtils::getRequestSkeleton(WF_KULCSLEKERDEZES, $variables);
        
        /* paramterek beillesztse */
        WebShopXmlUtils::addParameter($dom, $variables, CLIENTCODE, CLIENTCODE_VALUE);
        WebShopXmlUtils::addParameter($dom, $variables, POSID, $posId);
        WebShopXmlUtils::addParameter($dom, $variables, MUVELET, $muvelet);

        /* alrs kiszmtsa s paramterknt beszrsa */
        $signatureFields = array(0 => $posId, $muvelet);
        $signatureText = SignatureUtils::getSignatureText($signatureFields);

        $pkcs8PrivateKey = SignatureUtils::loadPrivateKey($this->getPrivKeyFileName($this->property, $posId));
        $signature = SignatureUtils::generateSignature($signatureText, $pkcs8PrivateKey, $this->property, $this->logger);

        $attrName = null;
		$attrValue = null;
		
		if (version_compare(PHP_VERSION, '5.4.8', '>=')) {
			$attrName = 'algorithm';
			$attrValue = 'SHA512';
		}
		
        WebShopXmlUtils::addParameter($dom, $variables, CLIENTSIGNATURE, $signature, $attrName, $attrValue);

        $this->lastInputXml = WebShopXmlUtils::xmlToString($dom);
        $this->logger->debug($this->operationLogNames['kulcsLekerdezes'] . " keres:\n" . $this->lastInputXml);
		
        /* a folyamat indtsa */
        $workflowState = SoapUtils::startWorkflowSynch(WF_KULCSLEKERDEZES, $this->lastInputXml, $this->soapClient, $this->logger);
        $response = new WResponse(WF_KULCSLEKERDEZES, $workflowState);

        /* a folyamat vlasznak naplzsa */
        if ($response->isFinished()) {
            $this->logger->info($this->operationLogNames['kulcsLekerdezes'] . ' folyamat azonosito: ' 
                . $response->getInstanceId());
            $responseDom = $response->getResponseDOM();
            $this->lastOutputXml = WebShopXmlUtils::xmlToString($responseDom);
            $this->logger->debug($this->operationLogNames['kulcsLekerdezes'] . " valasz:\n" 
                . trim($this->lastOutputXml));
        }
        else {
            $this->logger->error($this->operationLogNames['kulcsLekerdezes'] . ' hiba!');
            $this->logger->error($workflowState);            
        }

        $this->logger->debug($this->operationLogNames['kulcsLekerdezes'] . ' befejezodott.');

        return $response;
    }
    
    function removeSensitiveParams($inputXml) {

      $cvcPattern = '/(.*)(<' . CVCCVV . '>)(\d+)(<\/' . CVCCVV . '>)(.*)/i';
      $cvcReplacement = '$1$2***$4$5';
      $expDatePattern = '/(.*)(<' . EXPIRATIONDATE . '>)(\d+)(<\/' . EXPIRATIONDATE . '>)(.*)/i';
      $expDateReplacement = '$1$2****$4$5';

      $result = preg_replace(array($cvcPattern,$expDatePattern), array($cvcReplacement, $expDateReplacement), $inputXml);

      return $result;

     }

}

?>