unit uXmlTools;

interface

uses
  XMLDoc, XMLIntf, Classes, Contnrs, SysUtils, Rio, SOAPHTTPClient,
  uMWAccess;

const
  DF_RESULTDATE: string = 'yyyy.mm.dd hh:nn:ss zzz';
  DF_INSTANCEIDGENERATOR: string = 'yyyymmddhhnnsszzz';

type TInputXml = class
  private
    fDoc: IXMLDocument;
    fVariablesTag: IXMLNode;
    fWorkflowName: string;

  public
    property doc: IXMLDocument read fDoc;
    property workflowName: string read fWorkflowName;
    property variablesTag: IXMLNode read fVariablesTag;
  public
    constructor create(workflowName: string; aOwner: TComponent); overload;

    {: Tranzakcis input xml sszelltsa }
    constructor create(workflowName: string;
                       valueList: TStrings;
                       aOwner: TComponent;
                       transformator: TStrings = nil); overload;

    procedure addTextVariable(varName: string; varValue: string);

end;

type TOutputXml = class
  private
    fDoc: IXMLDocument;
    fMessageList: TStrings;
    fRecordList: IXmlNodeList;
    fSiker: boolean;

  public
    property doc: IXMLDocument read fDoc;
    property messageList: TStrings read fMessageList;
    property recordList: IXmlNodeList read fRecordList;
    property siker: boolean read fSiker;

  public
    constructor create(result: array of byte;
                       aOwner: TComponent); overload;
    constructor create(result: string;
                       aOwner: TComponent); overload;

  private
    constructor create(result: TStream;
                       aOwner: TComponent); overload;

end;

function getMWAccessResultFromFile(workflowName: string): WorkflowState;

implementation

uses
  uConfig,
  Types,
  Forms,
  QControls,
  DateUtils,
  Windows;

{ TInputXml }

constructor TInputXml.create(workflowName: string; aOwner: TComponent);
var
  root: IXMLNode;
  templateName: IXMLNode;
begin
  self.fWorkflowName := workflowName;

  fDoc := NewXMLDocument();
  fDoc.Encoding := 'ISO-8859-2';
  fDoc.Options := fDoc.Options + [doNodeAutoIndent];

  root := doc.AddChild('StartWorkflow');
  templateName := root.AddChild('TemplateName');
  templateName.Text := workflowName;
  fVariablesTag := root.AddChild('Variables');
end;

procedure TInputXml.addTextVariable(varName, varValue: string);
begin
  variablesTag.AddChild(varName).Text := varValue;
end;

constructor TInputXml.create(
        workflowName: string;
        valueList: TStrings;
        aOwner: TComponent;
        transformator: TStrings);
var
  i: integer;
  varName: string;
begin
  create(workflowName, aOwner);
  for i:=0 to valueList.Count-1 do begin
     varName := valueList.Names[i];
     if (varName <> '') then begin
       if (transformator <> nil)
            and (transformator.IndexOfName(varName) > -1) then begin
          varName := transformator.Values[varName];
       end;
       self.addTextVariable(varName, valueList.ValueFromIndex[i]);
     end;
  end;
end;

{ TOutputXml }

constructor TOutputXml.create(result: string; aOwner: TComponent);
var
  ss: TStringStream;
begin
  ss := TStringStream.Create(result);
  create(ss, aOwner);
  ss.Free();
end;

constructor TOutputXml.create(result: array of byte; aOwner: TComponent);
var
  bs: TMemoryStream;
begin
  bs := TMemoryStream.create();
  bs.Write(result, length(result));
  create(bs, aOwner);
  bs.Free();
end;

constructor TOutputXml.create(result: TStream; aOwner: TComponent);
var
  s: WideString;
  bs: TMemoryStream;
  msgListNode: IXmlNode;
  recordNodes: IXMLNodeList;
  i: integer;
begin
  fDoc := NewXMLDocument();
  fDoc.Options := fDoc.Options + [doNodeAutoIndent];
  fDoc.LoadFromStream(result);

  fMessageList := TStringList.create();

  msgListNode := fDoc.DocumentElement.ChildNodes['messagelist'];
  for i:=0 to msgListNode.ChildNodes.Count-1 do begin
    if msgListNode.ChildNodes[i].IsTextElement then begin
      fMessageList.Add(msgListNode.ChildNodes[i].NodeValue);
    end;
  end;

  fSiker := (fMessageList.count = 1) and (fMessageList[0] = 'SIKER');

  fRecordList := fDoc.DocumentElement.ChildNodes['resultset'].ChildNodes;
end;

function getMWAccessResultFromFile(workflowName: string): WorkflowState;
var
  generatedInstanceId: string;
  startTime: TDateTime;
  inputStream: TFileStream;
  fakeXmlPath: string;
  xmlArray: array of byte;
  xmlArrayPtr: pointer;
  tmp: string;
begin
  result := WorkflowState.Create();
  result.completed := true;
  result.timeout := false;
  result.templateName := workflowName;

  fakeXmlPath :=
    getDemoXmlDir() + '/' + workflowName + '_output.xml';
  inputStream := TFileStream.Create(fakeXmlPath, fmOpenRead, fmShareDenyNone);
  SetLength(xmlArray, inputStream.Size);
  xmlArrayPtr := @xmlArray[0];
  inputStream.ReadBuffer(xmlArrayPtr^, inputStream.Size);
  inputStream.Free();
  result.result := TByteDynArray(xmlArray);

  DateTimeToString(tmp, DF_RESULTDATE, now);
  result.startTime := tmp;
  result.endTime := tmp;

  DateTimeToString(tmp, DF_INSTANCEIDGENERATOR, now);
  result.instanceId := tmp;
end;

end.

