package hu.iqsoft.otp.webshop;

import hu.iqsoft.otp.webshop.model.WebShopFizetesAdatokLista;
import hu.iqsoft.otp.webshop.model.WebShopFizetesValasz;
import hu.iqsoft.otp.webshop.model.WebShopTranzAzon;
import hu.iqsoft.otp.webshop.mwaccess.WorkflowState;
import hu.iqsoft.otp.webshop.util.SignatureUtils;
import hu.iqsoft.otp.webshop.util.TransactionLogger;
import hu.iqsoft.otp.webshop.util.WebShopXmlUtils;
import hu.iqsoft.otp.workflow.wlpi.WResponse;

import java.io.File;
import java.io.FileInputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;

import org.apache.axis.AxisFault;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

/**
 * Kliens oldali WebShop szerver szimultor.
 *
 * A WebShop-ok rszre ksztett, WebService felleten hvhat web-alkalmazs
 * rsze. A web-alkalmazst lokkisan, WebShop-onknt kell telepteni. Ez
 * az alkalmazs csupn szimullja az OTP Bank rendszernek megfelel WebShop
 * folyamat-hvst: 
 * - ping 
 * - tranzakci azonost generls 
 * - hromszerepls fizetsi folyamat indtsa 
 * - ktszerepls fizetsi folyamat indtsa 
 * - tranzakci adatok, tranzakci sttusz lekrdezse
 * - ktlpcss fizets lezrs
 *
 * A fenti szolgltatsok az OTP MWAccess fellet hvst szimulljk, az egyes
 * krsekre adand vlaszokat fjlbl olvassk ki.
 * A fjlok elnevezsi konvencijnak a [tranzakcinev]_output.xml kell lenni,
 * ahol a [tranzakcinev] lehet WEBSHOPTRANZAZONGENERALAS, 
 * WEBSHOPTRANZAKCIOLEKERDEZES, WEBSHOPFIZETES, WEBSHOPFIZETESKETSZEREPLOS 
 * s WEBSHOPFIZETESLEZARAS.  
 * A knyvtrat, melyben ezek a fjlok elhelyezend?k, a konfigurcis llomny
 * otp.webshop.SIMULATED_XML_DIR bejegyzsben kell megadni, kvetve a szoksos
 * elrsi tvonal megadsi szablyokat.
 * 
 * @version 1.0
 * @author Bodnr Imre (c) IQSYS Rt.
 */
public class WebShopServiceSimulator extends WebShopService {

    public static final String PROPERTY_SIMULATEDXMLDIR = "otp.webshop.SIMULATED_XML_DIR"; 

    /**
    * Konstruktor. Beolvassa a property fjlot s az alapjn inicializlja
    * a mw kapcsolatot s a ssl belltsokat
    */
   public WebShopServiceSimulator() throws org.apache.axis.AxisFault {
       this(null);
       try {
           category.debug("OTPWebShopServices *** SZIMULATOR *** peldanyositas...");

           category.debug("OTPWebShopService *** SZIMULATOR ***  peldanyositas befejezve...");
       } catch (Exception e) {
           category.fatal(LOG_FATALERROR, e);
           e.printStackTrace();
           throw new AxisFault(LOG_FATALERROR, e.getMessage(), null, null);
       }
  }

  public WebShopServiceSimulator(javax.xml.rpc.Service service) throws org.apache.axis.AxisFault {
      super(service);
  }

  protected WorkflowState getSimulatedResult(String workflowName) throws AxisFault {
      WorkflowState result = new WorkflowState();
      
      SimpleDateFormat timeSdf = new SimpleDateFormat("yyyy.MM.dd HH.mm.ss SSS");
      SimpleDateFormat instanceIdSdf = new SimpleDateFormat("yyMMddHHmmssSSS");
      Date now = new Date();
      
      result.setCompleted(true);
      result.setTimeout(false);
      result.setStartTime(timeSdf.format(now));
      result.setEndTime(timeSdf.format(now));
      result.setInstanceId(instanceIdSdf.format(now));
      result.setTemplateName(workflowName);

      try {
          Properties properties = getWebShopProperties();
          String simXmlFolder = properties.getProperty(PROPERTY_SIMULATEDXMLDIR);
          File simXmlFile = new File(simXmlFolder + File.separator + workflowName + "_output.xml");
          FileInputStream fis = new FileInputStream(simXmlFile);
          
          byte[] resultXml = new byte[(int)simXmlFile.length()];
          for (int i = 0; i < resultXml.length; i += fis.read(resultXml, i, resultXml.length - i));
          
          result.setResult(resultXml);
          
      } catch (Exception e) {
          category.fatal(LOG_FATALERROR, e);
          e.printStackTrace();
          throw new AxisFault(LOG_FATALERROR, e.getMessage(), null, null);
      }
      
      return result;
  }
  
  
  
  /**
   * Ping MWAccess service szimulci
   * Digitlis alrs nem kpzdik.
   */
  public void ping() throws AxisFault  {
    try {
        category.debug("ping *** SZIMULATOR *** indul...");
        category.debug("ping *** SZIMULATOR *** befejezdtt.");
    } catch (Exception e) {
        category.warn(SERVERERROR_INFO, e);
        throw new AxisFault("", e.getMessage(), null, null);
    }
  }

  /**
   * WEBSHOPTRANZAZONGENERALAS folyamat szinkron indts szimulci
   * Digitlis alrs nem kpzdik.
   */
  public WebShopTranzAzon tranzakcioAzonositoGeneralas(String posId) throws AxisFault {
    try {
        WebShopTranzAzon result = null;

        category.debug("tranzakcioAzonositoGeneralas *** SZIMULATOR *** indul...");

        Document dom = WebShopXmlUtils.getRequestSkeleton();

        Node templateNameText = dom.createTextNode(WebShopXmlUtils.TEMPLATENAME_TAGNAME);
        templateNameText.setNodeValue("WEBSHOPTRANZAZONGENERALAS");

        Node templateName =
          dom.getDocumentElement().getElementsByTagName(WebShopXmlUtils.TEMPLATENAME_TAGNAME).item(0);
        templateName.appendChild(templateNameText);

        Node variables =
          dom.getDocumentElement().getElementsByTagName(WebShopXmlUtils.VARIABLES_TAGNAME).item(0);

		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.POSID, posId);

        // alrs kiszmtsa s paramterknt beszrsa
        Properties properties = getWebShopProperties();
        String privKeyFileName = properties.getProperty(PROPERTY_PRIVATEKEYFILE);
        byte[] pkcs8PrivateKey = SignatureUtils.loadPrivateKey(privKeyFileName);
        String[] fields = {posId};
        byte[] autorizaciosKodBytes = 
            SignatureUtils.generateSignature(
                SignatureUtils.getByteArray(fields), pkcs8PrivateKey);

        String autorizaciosKod = SignatureUtils.toHex(autorizaciosKodBytes, 0, (int)autorizaciosKodBytes.length);
        WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.CLIENTSIGNATURE, autorizaciosKod);

        // a folyamat indtsa loggozssal
        String domString = WebShopXmlUtils.DOMToXMLString (dom);
        category.debug("tranzakcioAzonositoGeneralas *** SZIMULATOR *** keres:\n" + domString);

        WorkflowState state = getSimulatedResult("WEBSHOPTRANZAZONGENERALAS");
        
        category.info("tranzakcioAzonositoGeneralas folyamat azonosio: " + state.getInstanceId());
        System.out.println("Start: WEBSHOPTRANZAZONGENERALAS *SIMULATOR* with " + state.getInstanceId() + " at " + new Date());

        WResponse response = getResponse(state.getResult(), "WEBSHOPTRANZAZONGENERALAS");

        // a folyamat vlasznak loggoza
        domString = WebShopXmlUtils.DOMToXMLString (response.getOriginalResponseDOM().getOwnerDocument());
        category.debug("tranzakcioAzonositoGeneralas valasz:\n" + domString);

        response.acquire(false, ANSWERFACTORY);

        if (response.isSuccessful()) {
          result = (WebShopTranzAzon)response.getAnswer().getData();
          result.setValaszKod("SIKER");
        }
        else {
          result = new WebShopTranzAzon();
          result.setValaszKod(getErrorsAsList(response));
        }

        category.debug("tranzakcioAzonositoGeneralas sikeresen befejezodott.");

        return result;

    } catch (Exception e) {
        category.warn(SERVERERROR_INFO, e);
        throw new AxisFault("", e);
    }

  }

  /**
   * WEBSHOPTRANZAKCIOLEKERDEZES folyamat szinkron indtsa
   * Digitlis alrs kpzdik.
   */
  public WebShopFizetesAdatokLista tranzakcioStatuszLekerdezes(
        String posId, 
        String azonosito,
        String maxRekordSzam,
        String idoszakEleje,
        String idoszakVege) throws AxisFault {
    try {
        WebShopFizetesAdatokLista result;

        category.debug("tranzakcioStatuszLekerdezes *** SZIMULATOR *** indul.");

        Document dom = WebShopXmlUtils.getRequestSkeleton();

        Node templateNameText = dom.createTextNode(WebShopXmlUtils.TEMPLATENAME_TAGNAME);
        templateNameText.setNodeValue("WEBSHOPTRANZAKCIOLEKERDEZES");

        Node templateName =
          dom.getDocumentElement().getElementsByTagName(WebShopXmlUtils.TEMPLATENAME_TAGNAME).item(0);
        templateName.appendChild(templateNameText);

        Node variables =
          dom.getDocumentElement().getElementsByTagName(WebShopXmlUtils.VARIABLES_TAGNAME).item(0);

        /* paramterek beillesztse */
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.POSID, posId);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.TRANSACTIONID, azonosito);
        WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.QUERYMAXRECORDS, maxRekordSzam);
        WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.QUERYSTARTDATE, idoszakEleje);
        WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.QUERYENDDATE, idoszakVege);

        // alrs kiszmtsa s paramterknt beszrsa
		Properties properties = getWebShopProperties();
		String privKeyFileName = properties.getProperty(PROPERTY_PRIVATEKEYFILE);
        byte[] pkcs8PrivateKey = SignatureUtils.loadPrivateKey(privKeyFileName);
        String[] fields = {posId, azonosito, maxRekordSzam, idoszakEleje, idoszakVege};
        byte[] autorizaciosKodBytes = 
            SignatureUtils.generateSignature(
                SignatureUtils.getByteArray(fields), pkcs8PrivateKey);

        String autorizaciosKod = SignatureUtils.toHex(autorizaciosKodBytes, 0, (int)autorizaciosKodBytes.length);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.CLIENTSIGNATURE, autorizaciosKod);

        // a folyamat indtsa loggozssal
        String domString = WebShopXmlUtils.DOMToXMLString (dom);
        category.debug("tranzakcioStatuszLekerdezes *** SZIMULATOR *** keres:\n" + domString);

        WorkflowState state = getSimulatedResult("WEBSHOPTRANZAKCIOLEKERDEZES");

        category.info("tranzakcioStatuszLekerdezes folyamat azonosio: " + state.getInstanceId());
        System.out.println("Finished: WEBSHOPTRANZAKCIOLEKERDEZES *SIMULATOR* with " + state.getInstanceId() + " at " + new Date());

        WResponse response = getResponse(state.getResult(), "WEBSHOPTRANZAKCIOLEKERDEZES");

        // a folyamat vlasznak loggoza
        domString = WebShopXmlUtils.DOMToXMLString (response.getOriginalResponseDOM().getOwnerDocument());
        category.debug("tranzakcioStatuszLekerdezes valasz:\n" + domString);

        response.acquire(false, ANSWERFACTORY);

        if (response.isSuccessful()) {
          result = (WebShopFizetesAdatokLista)response.getAnswer().getData();
          result.setValaszKod("SIKER");
        }
        else {
          result = new WebShopFizetesAdatokLista();
          result.setValaszKod(getErrorsAsList(response));
        }

        result.setPosId(posId);

        category.debug("tranzakcioStatuszLekerdezes *** SZIMULATOR *** sikeresen befejezodott.");

        return result;

    } catch (Exception e) {
        //e.printStackTrace();
        category.warn(SERVERERROR_INFO, e);
        throw new AxisFault("", e.getMessage(), null, null);
    }

  }

  /**
   * Hromszerepls fizetsi folyamat (WEBSHOPFIZETES) szinkron indtsnak
   * szimullsa
   * Digitlis alrs kpzdik.
   *
   * @param posId webshop azonost
   * @param tranzakcioAzonosito fizetsi tranzakci azonost
   * @param osszeg fizetend sszeg, (num, max. 13+2), opcionlis
   * tizedesponttal. <br /> 
   * Nulla is lehet, ha a regisztraltUgyfelId paramter ki van tltve, s az
   * ugyfelRegisztracioKell rtke igaz. gy kell ugyanis jelezni azt, hogy nem
   * tnyleges vsrlsi tranzakcit kell indtani, hanem egy gyfl
   * regisztrlst, vagyis az gyfl krtyaadatainak bekrst s eltrolst a
   * banki oldalon.
   * @param devizanem fizetend devizanem
   * @param nyelvkod a megjelentend vev oldali fellet nyelve
   * @param nevKell a megjelentend vev oldali felleten be kell krni a vev
   * nevt
   * @param orszagKell a megjelentend vev oldali felleten be kell krni a
   * vev cmnek "orszg rszt"
   * @param megyeKell a megjelentend vev oldali
   * felleten be kell krni a vev cmnek "megye rszt"
   * @param telepulesKell a megjelentend vev oldali felleten be kell
   * krni a vev cmnek "telepls rszt"
   * @param iranyitoszamKell a megjelentend vev oldali felleten be kell
   * krni  a vev cmnek "irnytszm rszt"
   * @param utcaHazszamKell a megjelentend vev oldali felleten be kell
   * krni a vev cmnek "utca/hzszm rszt"
   * @param mailCimKell a megjelentend vev oldali felleten be kellkrni a
   * vev e-mail cmt
   * @param kozlemenyKell a megjelentend vev oldali felleten fel kell
   * knlni a kzlemny megadsnak lehetsgt
   * @param vevoVisszaigazolasKell a tranzakci eredmnyt a vev oldalon
   *  meg
   * kell jelenteni (azaz nem a backURL-re kell irnytani)
   * @param ugyfelRegisztracioKell ha a regisztraltUgyfelId rtke nem res,
   * akkor megadja, hogy a megadott azonost jonnan regisztrland-e, vagy mr
   * regisztrlsra kerlt az OTP Internetes Fizet felletn. Elbbi esetben
   * a kliens oldali bngszben olyan fizet oldal fog megjelenni, melyen meg
   * kell adni az azonosthoz tartoz jelszt, illetve a krtyaadatokat.
   * Utbbi esetben csak az azonosthoz tartoz jelsz kerl beolvassra az
   * rtestsi cmen kvl.
   * Ha a regisztraltUgyfelId rtke res, a pamarter
   * rtke nem kerl felhasznlsra.
   * @param regisztraltUgyfelId az OTP fizetfelleten regisztrland vagy
   * regisztrlt  gyfl azonost kdja. 
   * @param shopMegjegyzes a webshop megjegyzse a tranzakcihoz a vev
   * rszre
   * @param backURL a tranzakci vgrehajtsa utn erre az internet cmre kell
   * irnytani a vev oldalon az gyfelet (ha a vevoVisszaigazolasKell hamis)
   * @param ketlepcsosFizetes megadja, hogy ktlpcss fizets indtand-e.
   * True rtk esetn a fizetsi tranzakci ktlpcss lesz, 
   * azaz a terhelend sszeg csupn zrolsra kerl, 
   * s gy is marad a bolt ltal indtott lezr tranzakci 
   * indtsig avagy a zrols elvlsig.
   * Az alaprtelmezett (res) rtk a false rtket jelli, teht
   * a szokvnyos egylpcss azonnali fizets trtnik.
   *                
   * @return a tranzakci adatai, mely tartalmazza a webshop ltal
   * megadott fizetsi adatokat, a vev ltal megadott adatokat, s a
   * tranzakci eredmnyt a tejests idejvel s a POS autorizcis kddal
   * egytt.
   */
  public WebShopFizetesAdatokLista fizetesiTranzakcio(
                        String posId,
                        String azonosito,
                        String osszeg,
                        String devizanem,
                        String nyelvkod,
                        boolean nevKell,
                        boolean orszagKell,
                        boolean megyeKell,
                        boolean telepulesKell,
                        boolean iranyitoszamKell,
                        boolean utcaHazszamKell,
                        boolean mailCimKell,
                        boolean kozlemenyKell,
                        boolean vevoVisszaigazolasKell,
                        boolean ugyfelRegisztracioKell,
                        String regisztraltUgyfelId,
                        String shopMegjegyzes,
                        String backURL,
                        Boolean ketlepcsosFizetes)
                    throws AxisFault {

    try {
        WebShopFizetesAdatokLista result;

        category.debug("fizetesiTranzakcio indul...");

        Document dom = WebShopXmlUtils.getRequestSkeleton();

        Node templateNameText = dom.createTextNode(WebShopXmlUtils.TEMPLATENAME_TAGNAME);
        templateNameText.setNodeValue("WEBSHOPFIZETES");

        Node templateName =
          dom.getDocumentElement().getElementsByTagName(WebShopXmlUtils.TEMPLATENAME_TAGNAME).item(0);
        templateName.appendChild(templateNameText);

        Node variables =
            dom.getDocumentElement().getElementsByTagName(WebShopXmlUtils.VARIABLES_TAGNAME).item(0);

        /* paramterek beillesztse */
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.POSID, posId);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.TRANSACTIONID, azonosito);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.AMOUNT, osszeg);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.EXCHANGE, devizanem);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.LANGUAGECODE, nyelvkod);

		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.NAMENEEDED, nevKell);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.COUNTRYNEEDED, orszagKell);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.COUNTYNEEDED, megyeKell);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.SETTLEMENTNEEDED, telepulesKell);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.ZIPCODENEEDED, iranyitoszamKell);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.STREETNEEDED, utcaHazszamKell);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.MAILADDRESSNEEDED, mailCimKell);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.NARRATIONNEEDED, kozlemenyKell);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.CONSUMERRECEIPTNEEDED, vevoVisszaigazolasKell);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.BACKURL, backURL);

		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.SHOPCOMMENT, shopMegjegyzes);
        
        WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.CONSUMERREGISTRATIONNEEDED, ugyfelRegisztracioKell);
        WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.CONSUMERREGISTRATIONID, regisztraltUgyfelId);

        WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.TWOSTAGED,
				Boolean.TRUE.equals(ketlepcsosFizetes));
        
        // alrs kiszmtsa s paramterknt beszrsa
		Properties properties = getWebShopProperties();
		String privKeyFileName = properties.getProperty(PROPERTY_PRIVATEKEYFILE);
        byte[] pkcs8PrivateKey = SignatureUtils.loadPrivateKey(privKeyFileName);
        String[] fields = {posId, azonosito, osszeg, devizanem, regisztraltUgyfelId};
        byte[] autorizaciosKodBytes = 
            SignatureUtils.generateSignature(
                SignatureUtils.getByteArray(fields), pkcs8PrivateKey);

        String autorizaciosKod = SignatureUtils.toHex(autorizaciosKodBytes, 0, (int)autorizaciosKodBytes.length);

		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.CLIENTSIGNATURE, autorizaciosKod);

        // a folyamat indtsa loggozssal
        String domString = WebShopXmlUtils.DOMToXMLString (dom);
        category.info("fizetesiTranzakcio *** SZIMULATOR *** keres:\n" + domString);

        /* Tranzakci adatainak naplzsa egy kln fjlba */
        TransactionLogger transLogger = 
            new TransactionLogger(
                properties.getProperty(PROPERTY_TRANSACTIONLOGDIR),
                category);

        transLogger.logHaromszereplosFizetesInditas(
            posId, azonosito, osszeg, devizanem, nyelvkod, nevKell,
            orszagKell, megyeKell, telepulesKell, iranyitoszamKell,
            utcaHazszamKell, mailCimKell, kozlemenyKell, vevoVisszaigazolasKell,
			ugyfelRegisztracioKell, regisztraltUgyfelId, shopMegjegyzes,
            backURL, ketlepcsosFizetes);

        WorkflowState state = getSimulatedResult("WEBSHOPFIZETES");

        category.info("fizetesiTranzakcio folyamat azonosio: " + state.getInstanceId());
        System.out.println("Finished: WEBSHOPFIZETES *SIMULATOR* with " + state.getInstanceId() + " at " + new Date());

        WResponse response = getResponse(state.getResult(), "WEBSHOPFIZETES");

        // a folyamat vlasznak loggoza
        domString = WebShopXmlUtils.DOMToXMLString (response.getOriginalResponseDOM().getOwnerDocument());
        category.debug("fizetesiTranzakcio *** SZIMULATOR *** valasz:\n" + domString);

        response.acquire(false, ANSWERFACTORY);

        if (response.isSuccessful()) {
          result = (WebShopFizetesAdatokLista)response.getAnswer().getData();
          result.setValaszKod("SIKER");
        }
        else {
          result = new WebShopFizetesAdatokLista();
          result.setValaszKod(getErrorsAsList(response));
        }

        result.setPosId(posId);

        transLogger.logHaromszereplosFizetesBefejezes(
            azonosito, result);
        
        category.debug("fizetesiTranzakcio *** SZIMULATOR *** befejezodott.");
        
        return result;

    } catch (Exception e) {
        //e.printStackTrace();
        category.warn(SERVERERROR_INFO, e);
        throw new AxisFault("", e.getMessage(), null, null);
    }
  }

  /**
   * Ktszerepls fizetsi tranzakci (WEBSHOPFIZETESKETSZEREPLOS) szimullsa.
   * Digitlis alrs kpzdik.
   *
   * @param posId tranzakci egyedi azonostja (alfanum, max. 32, azonos a 3
   * szereplsnl bevezetettel)
   * @param azonosito a shop azonostja (num, max. 6, azonos a 3 szereplsnl
   * bevezetettel)
   * @param osszeg vsrls sszege (num, max. 13+2), opcionlis tizedesponttal
   * @param devizanem vsrls devizaneme (opcionlis, azonos a 3 szereplsnl
   * bevezetettel)
   * @param nyelvkod nyelvkd (azonos a 3 szereplsnl bevezetettel)
   * @param regisztraltUgyfelId az OTP fizetfelleten regisztrlt gyfl
   * azonost kdja. Kitltse esetn a kartyaszam, cvc2cvv2, kartyaLejarat
   * adatok nem kerlnek feldolgozsra, hanem a banki oldalon az adott
   * azonosthoz eltrolt krtyaadatok kerlnek behelyettestsre
   * @param kartyaszam    krtyaszm (azonos a 3 szereplsnl bevezetettel)
   * @param cvc2cvv2      CVC2/CVV2 kd (azonos a 3 szereplsnl bevezetettel)
   * @param kartyaLejarat krtya lejrati dtuma, MMyy formban
   * @param vevoNev       vev neve (alfanum, max. 50, opcionlis, csak logozand)
   * @param vevoPostaCim  vev postai cme (alfanum, max. 100, opcionlis)
   * @param vevoIPCim     vev gpnek IP cme (alfanum, max. 15, opcionlis)
   * @param ertesitoMail  vev kirtestsi mailcme (alfanum, max. 50, opcionlis,
   *                      ha van, akkor mail kldend a tranzakci eredmnyrl
   *                      erre a cmre)
   * @param ertesitoTel   vev kirtestsi telefonszma (alfanum, max. 20,
   *                      opcionlis, ha van, akkor SMS kldend a tranzakci
   *                      eredmnyrl erre a telefonszmra)
   * @param ketlepcsosFizetes megadja, hogy ktlpcss fizets indtand-e.
   *                      True rtk esetn a fizetsi tranzakci ktlpcss lesz, 
   *                      azaz a terhelend sszeg csupn zrolsra kerl, 
   *                      s gy is marad a bolt ltal indtott lezr tranzakci 
   *                      indtsig avagy a zrols elvlsig.
   *                      Az alaprtelmezett (res) rtk a false rtket jelli, teht
   *                      a szokvnyos egylpcss azonnali fizets trtnik.   *                      
   */
  public WebShopFizetesValasz fizetesiTranzakcioKetszereplos(
                        String posId,
                        String azonosito,
                        String osszeg,
                        String devizanem,
                        String nyelvkod,
                        String regisztraltUgyfelId,
                        String kartyaszam,
                        String cvc2cvv2,
                        String kartyaLejarat,
                        String vevoNev,
                        String vevoPostaCim,
                        String vevoIPCim,
                        String ertesitoMail,
                        String ertesitoTel,
                        Boolean ketlepcsosFizetes)
                    throws AxisFault {
    try {
        WebShopFizetesValasz result;
        category.debug("fizetesiTranzakcioKetszereplos *** SZIMULATOR *** indul...");

        Document dom = WebShopXmlUtils.getRequestSkeleton();

        Node templateNameText = dom.createTextNode(WebShopXmlUtils.TEMPLATENAME_TAGNAME);
        templateNameText.setNodeValue("WEBSHOPFIZETESKETSZEREPLOS");

        Node templateName =
          dom.getDocumentElement().getElementsByTagName(WebShopXmlUtils.TEMPLATENAME_TAGNAME).item(0);
        templateName.appendChild(templateNameText);

        // default rtkek feldolgozsa
        if ((devizanem == null) || (devizanem.trim().length() == 0)) {
          devizanem = DEFAULT_DEVIZANEM;
        }

        Node variables =
          dom.getDocumentElement().getElementsByTagName(WebShopXmlUtils.VARIABLES_TAGNAME).item(0);

        /* paramterek beillesztse */
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.POSID, posId);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.TRANSACTIONID, azonosito);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.AMOUNT, osszeg);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.EXCHANGE, devizanem);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.LANGUAGECODE, nyelvkod);

        WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.CONSUMERREGISTRATIONID, regisztraltUgyfelId);

		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.CARDNUMBER, kartyaszam);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.CVCCVV, cvc2cvv2);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.EXPIRATIONDATE, kartyaLejarat);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.NAME, vevoNev);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.FULLADDRESS, vevoPostaCim);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.IPADDRESS, vevoIPCim);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.MAILADDRESS, ertesitoMail);
		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.TELEPHONE, ertesitoTel);
        WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.TWOSTAGED,
				Boolean.TRUE.equals(ketlepcsosFizetes));
        
        // alrs kiszmtsa s paramterknt beszrsa
        // ktszerepls esetben tbb mez szerepel az alrsnl...
        Properties properties = getWebShopProperties();
		String privKeyFileName = properties.getProperty(PROPERTY_PRIVATEKEYFILE);
        byte[] pkcs8PrivateKey = SignatureUtils.loadPrivateKey(privKeyFileName);
        String[] fields = {posId, azonosito, osszeg, devizanem, kartyaszam, cvc2cvv2, kartyaLejarat, regisztraltUgyfelId};
        byte[] autorizaciosKodBytes = 
            SignatureUtils.generateSignature(
                SignatureUtils.getByteArray(fields), pkcs8PrivateKey);

        String autorizaciosKod = SignatureUtils.toHex(autorizaciosKodBytes, 0, (int)autorizaciosKodBytes.length);

		WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.CLIENTSIGNATURE, autorizaciosKod);

        String domString = WebShopXmlUtils.DOMToXMLString (dom);
        category.info("fizetesiTranzakcioKetszereplos *** SZIMULATOR *** keres:\n" + domString);

        /* Tranzakci adatainak naplzsa egy kln fjlba */
        TransactionLogger transLogger = 
            new TransactionLogger(
                properties.getProperty(PROPERTY_TRANSACTIONLOGDIR),
                category);

        transLogger.logKetszereplosFizetesInditas(
            posId, azonosito, osszeg, devizanem, nyelvkod,
            regisztraltUgyfelId, kartyaszam, cvc2cvv2, kartyaLejarat,
            vevoNev, vevoPostaCim, vevoIPCim, ertesitoMail, ertesitoTel, ketlepcsosFizetes);
        
        WorkflowState state = getSimulatedResult("WEBSHOPFIZETESKETSZEREPLOS");

        category.info("fizetesiTranzakcioKetszereplos folyamat azonosio: " + state.getInstanceId());
        System.out.println("Finished: WEBSHOPFIZETESKETSZEREPLOS *SIMULATOR* with " + state.getInstanceId() + " at " + new Date());

        WResponse response = getResponse(state.getResult(), "WEBSHOPFIZETESKETSZEREPLOS");

        // a folyamat vlasznak loggoza
        domString = WebShopXmlUtils.DOMToXMLString (response.getOriginalResponseDOM().getOwnerDocument());
        category.debug("fizetesiTranzakcioKetszereplos valasz:\n" + domString);

        response.acquire(false, ANSWERFACTORY);

        if (response.isSuccessful()) {
          result = (WebShopFizetesValasz)response.getAnswer().getData();
          result.setValaszKod("SIKER");
        }
        else {
          result = new WebShopFizetesValasz();
          result.setValaszKod(getErrorsAsList(response));

          result.setPosId(posId);
          result.setAzonosito(azonosito);
        }


        transLogger.logKetszereplosFizetesBefejezes(
            azonosito, result);
        
        category.debug("fizetesiTranzakcioKetszereplos *** SZIMULATOR *** befejezodott.");

        return result;

    } catch (Exception e) {
        //e.printStackTrace();
        category.warn(SERVERERROR_INFO, e);
        throw new AxisFault("RENDSZERHIBA", e.getMessage(), null, null);
    }

  }

  /**
   * Hromszerepls fizetsi folyamat (WEBSHOPFIZETES) szinkron indtsa
   * Digitlis alrs kpzdik.
   *
   * @param posId
   *            webshop azonost
   * @param tranzakcioAzonosito
   *            fizetsi tranzakci azonost
   * @param jovahagyo
   * 			  megadja, hogy a lezrs jvhagy vagy tilt jelleg, 
   *            azaz a ktlpcss fizets sorn zrolt sszeg tnylegesen
   *            be kell-e terhelni a vev szmljn, avagy stornzni
   *            kell a ttelt.
   *
   * @return a tranzakci adatai, mely tartalmazza a webshop ltal megadott
   *         fizetsi adatokat, a vev ltal megadott adatokat, s a
   *         tranzakci eredmnyt a tejests idejvel s a POS autorizcis
   *         kddal egytt.
   */
  public WebShopFizetesValasz fizetesiTranzakcioLezaras(
  		String posId,
          String azonosito, 
          boolean jovahagyo) throws AxisFault {

      try {
      	WebShopFizetesValasz result;

          category.debug("fizetesiTranzakcioLezaras *SIMULATOR* indul...");

          Document dom = WebShopXmlUtils.getRequestSkeleton();

          Node templateNameText = dom
                  .createTextNode(WebShopXmlUtils.TEMPLATENAME_TAGNAME);
          templateNameText.setNodeValue(WebShopXmlUtils.WF_KETLEPCSOSFIZETESLEZARAS);

          Node templateName = dom.getDocumentElement().getElementsByTagName(
                  WebShopXmlUtils.TEMPLATENAME_TAGNAME).item(0);
          templateName.appendChild(templateNameText);

          Node variables = dom.getDocumentElement().getElementsByTagName(
                  WebShopXmlUtils.VARIABLES_TAGNAME).item(0);

          /* paramterek beillesztse */
          WebShopXmlUtils.addParameter(dom, variables, WebShopXmlUtils.POSID,
                  posId);
          WebShopXmlUtils.addParameter(dom, variables,
                  WebShopXmlUtils.TRANSACTIONID, azonosito);
          WebShopXmlUtils.addParameter(dom, variables,
                  WebShopXmlUtils.APPROVED, jovahagyo);
          
          // alrs kiszmtsa s paramterknt beszrsa
          byte[] pkcs8PrivateKey = SignatureUtils
                  .loadPrivateKey(getPrivKeyFileName(getWebShopProperties(),
                          posId));
          String[] fields = { posId, azonosito };
          byte[] autorizaciosKodBytes = SignatureUtils.generateSignature(
                  SignatureUtils.getByteArray(fields), pkcs8PrivateKey);

          String autorizaciosKod = SignatureUtils.toHex(autorizaciosKodBytes,
                  0, (int) autorizaciosKodBytes.length);

          WebShopXmlUtils.addParameter(dom, variables,
                  WebShopXmlUtils.CLIENTSIGNATURE, autorizaciosKod);

          // a folyamat indtsa loggozssal
          String domString = WebShopXmlUtils.DOMToXMLString(dom);
          category.info("fizetesiTranzakcioLezaras *SIMULATOR* keres:\n" + domString);

          /* Tranzakci adatainak naplzsa egy kln fjlba */
          TransactionLogger transLogger = new TransactionLogger(
                  getTranLogDir(getWebShopProperties(), posId), category);

          transLogger.logFizetesLezarasInditas(posId, azonosito, jovahagyo);

          WorkflowState state = getSimulatedResult(WebShopXmlUtils.WF_KETLEPCSOSFIZETESLEZARAS);

          category.info("fizetesiTranzakcioLezaras *SIMULATOR* folyamat azonosio: "
                  + state.getInstanceId());
          System.out.println(
          		"Finished: " 
          		+ WebShopXmlUtils.WF_KETLEPCSOSFIZETESLEZARAS 
          		+ " *SIMULATOR* with "
                  + state.getInstanceId() + " at " + new Date());

          WResponse response = getResponse(
          		state.getResult(),
          		WebShopXmlUtils.WF_KETLEPCSOSFIZETESLEZARAS);

          // a folyamat vlasznak loggoza
          domString = WebShopXmlUtils.DOMToXMLString(response
                  .getOriginalResponseDOM().getOwnerDocument());
          category.debug("fizetesiTranzakcioLezaras *SIMULATOR* valasz:\n" + domString);

          response.acquire(false, ANSWERFACTORY);

          if (response.isSuccessful()) {
              result = (WebShopFizetesValasz) response.getAnswer()
                      .getData();
              result.setValaszKod("SIKER");
          } else {
              result = new WebShopFizetesValasz();
              result.setValaszKod(getErrorsAsList(response));
          }

          result.setPosId(posId);

          transLogger.logFizetesLezarasBefejezes(azonosito, result);

          category.debug("fizetesiTranzakcioLezaras *SIMULATOR* befejezodott.");

          return result;

      } catch (Exception e) {
          // e.printStackTrace();
          category.warn(SERVERERROR_INFO, e);
          throw new AxisFault("", e.getMessage(), null, null);
      }
  }
  
}
