package hu.iqsoft.otp.webshop.util;

import hu.iqsoft.otp.webshop.model.WebShopFizetesAdatok;
import hu.iqsoft.otp.webshop.model.WebShopFizetesAdatokLista;
import hu.iqsoft.otp.webshop.model.WebShopFizetesValasz;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.apache.log4j.Category;

/**
 * hu.iqsys.otp.webshop.util.TransactionLogger
 * Tranzakcinknti napl fjl ksztse a hrom- s ktszerepls
 * fizetsi tranzakcikhoz.
 * 
 * @author Bodnr Imre (c) IQSys Rt.
 */
public class TransactionLogger {

    private String logDir;
    private Category category;
    private String logFileName;
    
    private static String LOG_DATE_FORMAT = "yyyy.MM.dd. HH:mm:ss"; 
    private static String LOG_EMPTY_VALUE = "";
    
    /**
     * Konstruktor
     */
    public TransactionLogger(String logDir, Category logCategory) {
        this.logDir = (logDir == null ? "" : logDir.trim());
        this.category = logCategory;
    }
    
    /**
     * A tranzakcis log llomny nevnek s elrsi tvonalnak
     * meghatrozsa. Az llomny neve utal a tranzakci azonostjra
     * s az indt bolt azonostjra. 
     * Ha tranzakci indtsrl van sz, j fjlnv kerl kpzsre, 
     * esetleges _x postfix generlsval, ahol x egsz szm.
     * Ha tranzakci befejezdsrl van sz, akkor a tranzakci 
     * indtshoz tartoz adatokat tartalmaz fjl neve kerl meghatrozsra. 
     * 
     * @param azonosito fizetsi tranzakci azonost
     * @param posId	shopId
     * @param inditas igaz, ha a fizetsi tranzakci indtsnak
     * 		          naplzsrl van sz	
     * @return
     */
    private String getLogFileName(
    		String azonosito, 
            String posId, 
            boolean inditas) {
        
    	/* Knyvtr lrtehozsa, ha szksges */
        File logFileDir = new File(logDir);
        if (logFileDir.exists()) {
        	if (!logFileDir.isDirectory()) {
        		category.warn("Ervenytelen tranzakcio log konyvtar: "
        		    + logDir);
            }
        }
        else {
            category.warn(
                "A tranzakcio log konyvtar nem letezik: "
            		+ logDir);

            boolean success = logFileDir.mkdirs();
            if (!success) {
                category.warn(
                    "A tranzakcio log konyvtar nem hozhato letre: "
                		+ logDir);
            }            
        }
        
        String logFileName = 
            logDir + File.separator + posId + "_" + azonosito + ".log";
        
        // Fel kell kszlni arra, hogy az adott nven mr ltezik fjl
        File logFile = new File(logFileName);
        int i = 0;
        while (inditas && logFile.exists()) {
        	i++;
            logFile = new File(logFileName + "_" + i);
        }

        return logFile.getAbsolutePath();
    }
    
    /**
     * Objektum string reprezentlsa.
     * Annyiban tr el a toString() ltal visszaadott adattl, hogy
     * null rtk esetn res string a visszatrsi rtk, s nem
     * a "null" szveg
     * 
     * @param value rtk
     * @return string reprezentci
     */
    private String nvl(Object value) {
        return (value == null ? LOG_EMPTY_VALUE : value.toString());
    }
   
  /**
   * Hromszerepls fizetsi tranzakci indtsnak naplzsa.
   *
   * @param posId webshop azonost
   * @param azonosito fizetsi tranzakci azonost
   * @param osszeg fizetend sszeg 
   * @param devizanem fizetend devizanem
   * @param nyelvkod a megjelentend vev oldali fellet nyelve
   * @param nevKell a megjelentend vev oldali felleten be kell krni a vev
   * nevt
   * @param orszagKell a megjelentend vev oldali felleten be kell krni a
   * vev cmnek "orszg rszt"
   * @param megyeKell a megjelentend vev oldali
   * felleten be kell krni a vev cmnek "megye rszt"
   * @param telepulesKell a megjelentend vev oldali felleten be kell
   * krni a vev cmnek "telepls rszt"
   * @param iranyitoszamKell a megjelentend vev oldali felleten be kell
   * krni  a vev cmnek "irnytszm rszt"
   * @param utcaHazszamKell a megjelentend vev oldali felleten be kell
   * krni a vev cmnek "utca/hzszm rszt"
   * @param mailCimKell a megjelentend vev oldali felleten be kellkrni a
   * vev e-mail cmt
   * @param kozlemenyKell a megjelentend vev oldali felleten fel kell
   * knlni a kzlemny megadsnak lehetsgt
   * @param vevoVisszaigazolasKell a tranzakci eredmnyt a vev oldalon
   *  meg kell jelenteni (azaz nem a backURL-re kell irnytani)
   * @param ugyfelRegisztracioKell ha a regisztraltUgyfelId rtke nem res,
   * akkor megadja, hogy a megadott azonost jonnan regisztrland-e, vagy mr
   * regisztrlsra kerlt az OTP Internetes Fizet felletn. 
   * @param regisztraltUgyfelId az OTP fizetfelleten regisztrland vagy
   * regisztrlt  gyfl azonost kdja. 
   * @param shopMegjegyzes a webshop megjegyzse a tranzakcihoz a vev
   * rszre
   * @param backURL a tranzakci vgrehajtsa utn erre az internet cmre kell
   * irnytani a vev oldalon az gyfelet (ha a vevoVisszaigazolasKell hamis)
   */
    public void logHaromszereplosFizetesInditas(
            String posId,
            String azonosito,
            String osszeg,
            String devizanem,
            String nyelvkod,
            boolean nevKell,
            boolean orszagKell,
            boolean megyeKell,
            boolean telepulesKell,
            boolean iranyitoszamKell,
            boolean utcaHazszamKell,
            boolean mailCimKell,
            boolean kozlemenyKell,
            boolean vevoVisszaigazolasKell,
            boolean ugyfelRegisztracioKell,
            String regisztraltUgyfelId,
            String shopMegjegyzes,
            String backURL,
            Boolean ketlepcsosFizetes) {
    	
       if ((azonosito != null) && (azonosito.trim().length() > 0)) {
            logFileName = getLogFileName(azonosito, posId, true); 
        	try {
                PrintStream ps = 
                    new PrintStream(
                    	new FileOutputStream(logFileName), 
                    	false, 
                    	"ISO-8859-2");

                ps.println("Haromszereplos fizetesi tranzakcio");
                ps.println("\nInditas: " 
                    + new SimpleDateFormat(LOG_DATE_FORMAT).format(new Date()));
                
                ps.println("\nIndito adatok");
                ps.println("  posId: " + nvl(posId));
                ps.println("  azonosito: " + nvl(azonosito));
                ps.println("  osszeg: " + nvl(osszeg));
                ps.println("  devizanem: " + nvl(devizanem));
                ps.println("  nyelvkod: " + nvl(nyelvkod));
                ps.println("  nevKell: " + nevKell);
                ps.println("  orszagKell: " + orszagKell);
                ps.println("  megyeKell: " + megyeKell);
                ps.println("  telepulesKell: " + telepulesKell);
                ps.println("  iranyitoszamKell: " + iranyitoszamKell);
                ps.println("  utcaHazszamKell: " + utcaHazszamKell);
                ps.println("  mailCimKell: " + mailCimKell);
                ps.println("  kozlemenyKell: " + kozlemenyKell);
                ps.println("  vevoVisszaigazolasKell: " + vevoVisszaigazolasKell);
                ps.println("  ugyfelRegisztracioKell: " + ugyfelRegisztracioKell);
                ps.println("  regisztraltUgyfelId: " + nvl(regisztraltUgyfelId));
                ps.println("  shopMegjegyzes: " + nvl(shopMegjegyzes));
                ps.println("  backURL: " + nvl(backURL));
                ps.println("  ketlepcsosFizetes: " + 
                		(ketlepcsosFizetes == null ? LOG_EMPTY_VALUE : ketlepcsosFizetes.toString()) );	

                ps.flush();
                ps.close();
            } catch (IOException e) {
				category.warn("Hiba tortent a tranzakcios naplo fajl letrehozasa " 
                                + "vagy irasa kozben: " + logFileName);
			}
        }
        else {
        	category.warn("A tranzakcio adatai nem naplozhatoak," 
                            + "' az azonosito nincs megadva.");
            
        }
    }

   /**
    * Hromszerepls fizetsi tranzakci befejezdsnek naplzsa.
    *
    * @param azonosito fizetsi tranzakci azonost
    * @param valasz a fizetsi tranzakci vlasza
    */
    public void logHaromszereplosFizetesBefejezes(
            String azonosito,
    		WebShopFizetesAdatokLista valasz) {

        if ((azonosito != null) && (azonosito.trim().length() > 0)) {
            try {
                PrintStream ps = 
                    new PrintStream(
                    	new FileOutputStream(logFileName, true), 
                    	true, 
                    	"ISO-8859-2");

                ps.println("\nBefejezes: " 
                    + new SimpleDateFormat(LOG_DATE_FORMAT).format(new Date()));
                
                ps.println("\nValasz kod: " + valasz.getValaszKod());

                if (valasz.getWebShopFizetesAdatok() != null
                        && valasz.getWebShopFizetesAdatok().length > 0) {
                    WebShopFizetesAdatok fizetesAdatok =
                        valasz.getWebShopFizetesAdatok()[0];

                    ps.println("\nValasz adatok");
                    ps.println("  posId: " + nvl(fizetesAdatok.getPosId()));
                    ps.println("  azonosito: " + nvl(fizetesAdatok.getAzonosito()));
                    ps.println("  posValaszkod: " + nvl(fizetesAdatok.getPosValaszkod()));
                    ps.println("  autorizaciosKod: " + nvl(fizetesAdatok.getAutorizaciosKod()));
                    ps.println("  statuszKod: " + nvl(fizetesAdatok.getStatuszKod()));
                    ps.println("  teljesites: " + nvl(fizetesAdatok.getTeljesites()));
                    ps.println("  nev: " + nvl(fizetesAdatok.getNev()));
                    ps.println("  orszag: " + nvl(fizetesAdatok.getOrszag()));
                    ps.println("  megye: " + nvl(fizetesAdatok.getMegye()));
                    ps.println("  varos: " + nvl(fizetesAdatok.getVaros()));
                    ps.println("  iranyitoszam: " + nvl(fizetesAdatok.getIranyitoszam()));
                    ps.println("  utcaHazszam: " + nvl(fizetesAdatok.getUtcaHazszam()));
                    ps.println("  mailCim: " + nvl(fizetesAdatok.getMailCim()));
                    ps.println("  kozlemeny: " + nvl(fizetesAdatok.getKozlemeny()));
                }

                ps.flush();
                ps.close();
            } catch (IOException e) {
                category.warn("Hiba tortent a tranzakcios naplo fajl letrehozasa " 
                                + "vagy irasa kozben: " + logFileName);
            }
        }
        else {
            category.warn("A tranzakcio adatai nem naplozhatoak," 
                            + "' az azonosito nincs megadva.");
            
        }
    }
    
   /**
    * Ktszerepls fizetsi tranzakci indtsi adatainak naplzsa.
    *
    * @param posId tranzakci egyedi azonostja 
    * @param azonosito a shop azonostja 
    * @param osszeg vsrls sszege 
    * @param devizanem vsrls devizaneme 
    * @param nyelvkod nyelvkd 
    * @param regisztraltUgyfelId az OTP fizetfelleten regisztrlt gyfl
    * azonost kdja. 
    * @param kartyaszam    krtyaszm 
    * @param cvc2cvv2      CVC2/CVV2 kd 
    * @param kartyaLejarat krtya lejrati dtuma, MMyy formban
    * @param vevoNev       vev neve 
    * @param vevoPostaCim  vev postai cme 
    * @param vevoIPCim     vev gpnek IP cme 
    * @param ertesitoMail  vev kirtestsi mailcme 
    * @param ertesitoTel   vev kirtestsi telefonszma 
    */
    public void logKetszereplosFizetesInditas(
            String posId,
            String azonosito,
            String osszeg,
            String devizanem,
            String nyelvkod,
            String regisztraltUgyfelId,
            String kartyaszam,
            String cvc2cvv2,
            String kartyaLejarat,
            String vevoNev,
            String vevoPostaCim,
            String vevoIPCim,
            String ertesitoMail,
            String ertesitoTel,
            Boolean ketlepcsosFizetes) {
        
        if ((azonosito != null) && (azonosito.trim().length() > 0)) {
            logFileName = getLogFileName(azonosito, posId, true);
            try {
                PrintStream ps = 
                    new PrintStream(
                    	new FileOutputStream(logFileName), 
                    	false, 
                    	"ISO-8859-2");

                ps.println("Ketszereplos fizetesi tranzakcio");
                ps.println("\nInditas: " 
                    + new SimpleDateFormat(LOG_DATE_FORMAT).format(new Date()));
                
                ps.println("\nIndito adatok");
                ps.println("  posId: " + nvl(posId));
                ps.println("  azonosito: " + nvl(azonosito));
                ps.println("  osszeg: " + nvl(osszeg));
                ps.println("  devizanem: " + nvl(devizanem));
                ps.println("  nyelvkod: " + nvl(nyelvkod));
                ps.println("  regisztraltUgyfelId: " + nvl(regisztraltUgyfelId));
                ps.println("  vevoNev: " + nvl(vevoNev));
                ps.println("  vevoPostaCim: " + nvl(vevoPostaCim));
                ps.println("  vevoIPCim: " + nvl(vevoIPCim));
                ps.println("  ertesitoMail: " + nvl(ertesitoMail));
                ps.println("  ertesitoTel: " + nvl(ertesitoTel));
                ps.println("  ketlepcsos: " + 
                		(ketlepcsosFizetes == null ? LOG_EMPTY_VALUE : ketlepcsosFizetes.toString()) );	
                
                ps.flush();
                ps.close();
            } catch (IOException e) {
                category.warn("Hiba tortent a tranzakcios naplo fajl letrehozasa " 
                                + "vagy irasa kozben: " + logFileName);
            }
        }
        else {
            category.warn("A tranzakcio adatai nem naplozhatoak," 
                            + "' az azonosito nincs megadva.");
            
        }
        
    }

   /**
    * Ktszerepls fizetsi tranzakci befejezdsnek naplzsa.
    *
    * @param azonosito fizetsi tranzakci azonost
    * @param valasz a fizetsi tranzakci vlasza
    */
    public void logKetszereplosFizetesBefejezes(
            String azonosito,
            WebShopFizetesValasz valasz) {

        if ((azonosito != null) && (azonosito.trim().length() > 0)) {
            try {
                PrintStream ps = 
                    new PrintStream(
                    	new FileOutputStream(logFileName, true), 
                    	true, 
                    	"ISO-8859-2");

                ps.println("\nBefejezes: " 
                    + new SimpleDateFormat(LOG_DATE_FORMAT).format(new Date()));
                
                ps.println("\nValasz kod: " + valasz.getValaszKod());

                ps.println("\nValasz adatok");
                ps.println("  posId: " + nvl(valasz.getPosId()));
                ps.println("  azonosito: " + nvl(valasz.getAzonosito()));
                ps.println("  posValaszkod: " + nvl(valasz.getValaszKod()));
                ps.println("  autorizaciosKod: " + nvl(valasz.getAuthorizaciosKod()));
                ps.println("  teljesites: " + nvl(valasz.getTeljesites()));
                
                ps.flush();
                ps.close();
            } catch (IOException e) {
                category.warn("Hiba tortent a tranzakcios naplo fajl letrehozasa " 
                                + "vagy irasa kozben: " + logFileName);
            }
        }
        else {
            category.warn("A tranzakcio adatai nem naplozhatoak," 
                            + "' az azonosito nincs megadva.");
            
        }
    }

    /**
     * Ktlpcss fizetsi tranzakci lezrsa indtsi adatainak naplzsa.
     *
     * @param posId tranzakci egyedi azonostja 
     * @param azonosito a shop azonostja 
     * @param jovahagyo jvhagy-e a lezrs
     */
     public void logFizetesLezarasInditas(
             String posId,
             String azonosito,
             boolean jovahagyo) {
         
         if ((azonosito != null) && (azonosito.trim().length() > 0)) {
             logFileName = getLogFileName(azonosito, posId, true);
             try {
                 PrintStream ps = 
                     new PrintStream(
                     	new FileOutputStream(logFileName), 
                     	false, 
                     	"ISO-8859-2");

                 ps.println("Ketlepcsos fizetes lezaras tranzakcio");
                 ps.println("\nInditas: " 
                     + new SimpleDateFormat(LOG_DATE_FORMAT).format(new Date()));
                 
                 ps.println("\nIndito adatok");
                 ps.println("  posId: " + nvl(posId));
                 ps.println("  azonosito: " + nvl(azonosito));
                 ps.println("  jovahagyo: " + jovahagyo);

                 ps.flush();
                 ps.close();
             } catch (IOException e) {
                 category.warn("Hiba tortent a tranzakcios naplo fajl letrehozasa " 
                                 + "vagy irasa kozben: " + logFileName);
             }
         }
         else {
             category.warn("A tranzakcio adatai nem naplozhatoak," 
                             + "' az azonosito nincs megadva.");
             
         }
         
     }

    /**
     * Ktlpcss fizetsi tranzakci lezrsa befejezdsnek naplzsa.
     *
     * @param azonosito fizetsi tranzakci azonost
     * @param valasz a fizetsi tranzakci vlasza
     */
     public void logFizetesLezarasBefejezes(
             String azonosito,
             WebShopFizetesValasz valasz) {

         if ((azonosito != null) && (azonosito.trim().length() > 0)) {
             try {
                 PrintStream ps = 
                     new PrintStream(
                     	new FileOutputStream(logFileName, true), 
                     	true, 
                     	"ISO-8859-2");

                 ps.println("\nBefejezes: " 
                     + new SimpleDateFormat(LOG_DATE_FORMAT).format(new Date()));
                 
                 ps.println("\nValasz kod: " + valasz.getValaszKod());

                 ps.println("\nValasz adatok");
                 ps.println("  posId: " + nvl(valasz.getPosId()));
                 ps.println("  azonosito: " + nvl(valasz.getAzonosito()));
                 ps.println("  posValaszkod: " + nvl(valasz.getValaszKod()));
                 ps.println("  autorizaciosKod: " + nvl(valasz.getAuthorizaciosKod()));
                 ps.println("  teljesites: " + nvl(valasz.getTeljesites()));
                 
                 ps.flush();
                 ps.close();
             } catch (IOException e) {
                 category.warn("Hiba tortent a tranzakcios naplo fajl letrehozasa " 
                                 + "vagy irasa kozben: " + logFileName);
             }
         }
         else {
             category.warn("A tranzakcio adatai nem naplozhatoak," 
                             + "' az azonosito nincs megadva.");
             
         }
     }
    
}
